#include "mem_sequence_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/mock_node.h>
#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestMemSequenceNode {

Y_UNIT_TEST_SUITE(MemSequenceNodeSuite) {

static TLogger logger({});

TMemSequenceNodePtr SingleChildSequence(TTickResult tickResult) {
    return new TMemSequenceNode({1, "title"}, {new TMockNode({1, "title"}, tickResult)});
}

Y_UNIT_TEST(TestWithOneFailedChild) {
    TTickResult expected = TNodeError{"test error message"};

    auto sequence = SingleChildSequence(expected);

    auto result = sequence->Tick(MockTickContext(logger));
    UNIT_ASSERT_EQUAL(expected, result);
}

Y_UNIT_TEST(TestWithOneSuccessChild) {
    ENodeStatus const goodStatuses[] = {ENodeStatus::SUCCESS, ENodeStatus::FAILURE, ENodeStatus::RUNNING};

    for (auto&& status : goodStatuses) {
        TTickResult expected(status);
        auto sequence = SingleChildSequence(expected);

        auto result = sequence->Tick(MockTickContext(logger));
        UNIT_ASSERT_EQUAL(expected, result);
    }
}

Y_UNIT_TEST(TesWithOneUnexpextedChildStatus) {
    TTickResult idleChild(ENodeStatus::IDLE);

    auto sequence = SingleChildSequence(idleChild);

    auto result = sequence->Tick(MockTickContext(logger));
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "unexpected status");
}

Y_UNIT_TEST(TestTwoChildrenPendingUntilTotalSuccess) {
    TMemSequenceNodePtr sequence = new TMemSequenceNode(
        {1, "title"}
        , {
            new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::SUCCESS))
            , new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::SUCCESS))
        }
    );

    {
        auto result = sequence->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::SUCCESS);
    }

    {
        auto result = sequence->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::SUCCESS);
    }
}

Y_UNIT_TEST(TestTwoChildrenFailOnFirstFailure) {
    TMemSequenceNodePtr sequence = new TMemSequenceNode(
        {1, "title"}
        , {
            new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::FAILURE))
            , new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::SUCCESS))
        }
    );

    {
        auto result = sequence->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::FAILURE);
    }

    {
        auto result = sequence->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::FAILURE);
    }
}

Y_UNIT_TEST(TestTwoChildrenFailSecodFail) {
    TMemSequenceNodePtr sequence = new TMemSequenceNode(
        {1, "title"}
        , {
            new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::SUCCESS))
            , new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::FAILURE))
        }
    );

    {
        auto result = sequence->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::FAILURE);
    }

    {
        auto result = sequence->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::FAILURE);
    }

    {
        auto result = sequence->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::FAILURE);
    }

    {
        auto result = sequence->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::FAILURE);
    }
}


Y_UNIT_TEST(TestTwoChildrenRunning) {
    TMemSequenceNodePtr sequence = new TMemSequenceNode(
        {1, "title"}
        , {
            new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::SUCCESS))
            , new TMockNode({1, "title"}, TNodeSuccess(ENodeStatus::RUNNING))
        }
    );

    {
        auto result = sequence->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::RUNNING);
    }

    {
        auto result = sequence->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::RUNNING);
    }

    {
        auto result = sequence->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::RUNNING);
    }

    {
        auto result = sequence->Tick(MockTickContext(logger));
        UNIT_ASSERT((bool) result);
        UNIT_ASSERT_EQUAL(result.Success(), ENodeStatus::RUNNING);
    }
}

}

} // namespace NInfra::NPodAgent::NTestMemSequenceNode


