#pragma once

#include "case_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/basic_composite_node.h>

#include <util/generic/serialized_enum.h>
#include <util/string/cast.h>
#include <util/system/type_name.h>

#include <type_traits>

namespace NInfra::NPodAgent {


/**
 * Static Switch
 *
 * Only the child with TEnum equal to the SwitchValue parameter
 * (or the default one) is set as the child of this node, the
 * others are deleted for the sake of consuming less memory.
 * TickImpl method returns TTickResult of its sole child
 */

template <class TEnum>
class TStaticSwitchNode;

template <class TEnum>
using TStaticSwitchNodePtr = TSimpleSharedPtr<TStaticSwitchNode<TEnum>>;

template <class TEnum>
class TStaticSwitchNode : public TBasicCompositeNode {
public:
    TStaticSwitchNode(
        const TBasicTreeNodeDescriptor& descriptor
        , TEnum switchValue
    )
        : TBasicCompositeNode(descriptor, {})
        , SwitchValue_(switchValue)
    {}

    virtual ENodeType GetType() const override final {
        return TStaticSwitchNode::NODE_TYPE;
    }

    virtual void SetChildren(TVector<TBasicTreeNodePtr>&& children) override final {
        THashMap<TEnum, TBasicTreeNodePtr> childrenByType;
        TBasicTreeNodePtr defaultChild = nullptr;

        for (const auto& child: children) {
            TCaseNode *caseNode = dynamic_cast<TCaseNode*>(child.Get());

            Y_ENSURE(caseNode, "all children of StaticSwitch node must have type TCaseNode");

            if (caseNode->IsDefault()) {
                Y_ENSURE(!defaultChild, "children array of StaticSwitch node have more than 1 default cases");
                defaultChild = child;
            } else {
                TEnum value = FromString(caseNode->GetCaseValue());
                Y_ENSURE(!childrenByType.contains(value), "children array for StaticSwitch node contains at least two Case nodes with the same value: " << value);
                childrenByType[value] = child;
            }
        }

        auto enumAllValues = GetEnumAllValues<TEnum>();
        for (const auto& name: enumAllValues) {
            Y_ENSURE(ToString(name) != TCaseNode::DEFAULT_CASE_VALUE, TStringBuilder() << "enum " << TypeName<TEnum>() << " has '" << TCaseNode::DEFAULT_CASE_VALUE << "' value");
        }

        Y_ENSURE(defaultChild || enumAllValues.size() == childrenByType.size(), "the number of cases, " << enumAllValues.size() << " is not equal to the number of children, " << childrenByType.size());

        children.clear();
        if (childrenByType.contains(SwitchValue_)) {
            children.push_back(childrenByType[SwitchValue_]);
        } else {
            children.push_back(defaultChild);
        }

        TBasicCompositeNode::SetChildren(std::move(children));
    }

private:
    virtual TTickResult TickImpl(TTickContextPtr tickContext) override final {
        if (GetChildren().size() != 1) {
            return TTickResult(TNodeError{
                TStringBuilder() << "TStaticSwitchNode should have 1 child, but has"
                                 << GetChildren().size()
            });
        }
        return GetChildren()[0]->Tick(tickContext);
    }

public:
    static constexpr const ENodeType NODE_TYPE = ENodeType::STATIC_SWITCH;

private:
    TEnum SwitchValue_;
};


} // namespace NInfra::NPodAgent
