#include "static_switch_node.h"
#include "switch_node_ut.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/mock_node.h>
#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestStaticSwitchNode {

using namespace NTest;

Y_UNIT_TEST_SUITE(StaticSwitchNodeSuite) {

static TLogger logger({});

TBasicTreeNodePtr NodeWithResult(TTickResult tickResult) {
    return new TMockNode({1, "title"}, tickResult);
}

TBasicTreeNodePtr SingleChildCase(TTickResult tickResult, const TString& enumValue) {
    TCaseNodePtr result = new TCaseNode({1, "title"}, enumValue);
    result->SetChildren({NodeWithResult(tickResult)});
    return result;
}

template <class TEnum>
TStaticSwitchNodePtr<TEnum> GetStaticSwitch(TVector<TBasicTreeNodePtr>&& children, TEnum childType) {
    TStaticSwitchNodePtr<TEnum> result = new TStaticSwitchNode<TEnum>({1, "title"}, childType);
    result->SetChildren(std::move(children));
    return result;
}

Y_UNIT_TEST(TestEverythingSuccessful) {
    TVector<TBasicTreeNodePtr> children = {
        SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "ok"}, ToString(TE_OK))
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "fail"}, ToString(TE_FAIL))
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "semi_fail"}, ToString(TE_SEMI_FAIL))
    };
    TStaticSwitchNodePtr<ETestEnum> node;
    UNIT_ASSERT_NO_EXCEPTION(node = GetStaticSwitch<ETestEnum>(std::move(children), TE_OK));
    UNIT_ASSERT_EQUAL(node->GetChildren().size(), 1);

    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_EQUAL((TTickResult(TNodeSuccess{ENodeStatus::SUCCESS, "ok"})), result);
}

Y_UNIT_TEST(TestWrongCaseChildren) {
    TTickResult condition = TNodeSuccess{ENodeStatus::SUCCESS, ToString(TE_FAIL)};

    TVector<TBasicTreeNodePtr> children = {
        SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "ok"}, ToString(TE_OK))
        , NodeWithResult(condition)
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "semi_fail"}, ToString(TE_SEMI_FAIL))
    };
    UNIT_ASSERT_EXCEPTION_CONTAINS(GetStaticSwitch<ETestEnum>(std::move(children), TE_OK), yexception, "all children of StaticSwitch node must have type TCaseNode");
}

Y_UNIT_TEST(TestWrongCaseValue) {
    TVector<TBasicTreeNodePtr> children = {
        SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "ok"}, ToString(TE_OK))
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "fail"}, "wrong_value")
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "semi_fail"}, ToString(TE_SEMI_FAIL))
    };
    UNIT_ASSERT_EXCEPTION(GetStaticSwitch<ETestEnum>(std::move(children), TE_FAIL), yexception);
}

Y_UNIT_TEST(TestSameCases) {
    TVector<TBasicTreeNodePtr> children = {
        SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "ok"}, ToString(TE_OK))
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "ok"}, ToString(TE_OK))
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "semi_fail"}, ToString(TE_SEMI_FAIL))
    };
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        GetStaticSwitch<ETestEnum>(std::move(children), TE_FAIL)
        , yexception
        , "children array for StaticSwitch node contains at least two Case nodes with the same value: " + ToString(TE_OK)
    );
}

Y_UNIT_TEST(TestDefaultInEnum) {
    TVector<TBasicTreeNodePtr> children = {
        SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "ok"}, ToString(EWithDefaultField::OK))
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "fail"}, ToString(EWithDefaultField::FAIL))
    };

    UNIT_ASSERT_EXCEPTION_CONTAINS(GetStaticSwitch<EWithDefaultField>(std::move(children), EWithDefaultField::OK), yexception, TStringBuilder() << " has '" << TCaseNode::DEFAULT_CASE_VALUE << "' value");
}

Y_UNIT_TEST(TestMoreThanOneDefault) {
    TVector<TBasicTreeNodePtr> children = {
        SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "ok"}, ToString(TE_OK))
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "fail"}, ToString(TE_FAIL))
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "ok"}, TCaseNode::DEFAULT_CASE_VALUE)
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "fail"}, TCaseNode::DEFAULT_CASE_VALUE)
    };
    UNIT_ASSERT_EXCEPTION_CONTAINS(GetStaticSwitch<ETestEnum>(std::move(children), TE_SEMI_FAIL), yexception, "more than 1 default cases");
}

Y_UNIT_TEST(TestCasesLessThanEnumsWithoutDefault) {
    TVector<TBasicTreeNodePtr> children = {
        SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "ok"}, ToString(TE_OK))
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "semi_fail"}, ToString(TE_SEMI_FAIL))
    };
    UNIT_ASSERT_EXCEPTION_CONTAINS(GetStaticSwitch<ETestEnum>(std::move(children), TE_OK), yexception, "is not equal to the number of children");
}

Y_UNIT_TEST(TestEverythingSuccessfulWithDefault) {
    TVector<TBasicTreeNodePtr> children = {
        SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "ok"}, ToString(TE_OK))
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "fail"}, ToString(TE_FAIL))
        , SingleChildCase(TNodeSuccess{ENodeStatus::SUCCESS, "semi_fail"}, TCaseNode::DEFAULT_CASE_VALUE)
    };
    TStaticSwitchNodePtr<ETestEnum> node;
    UNIT_ASSERT_NO_EXCEPTION(node = GetStaticSwitch<ETestEnum>(std::move(children), TE_SEMI_FAIL));
    UNIT_ASSERT_EQUAL(node->GetChildren().size(), 1);

    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_EQUAL((TTickResult(TNodeSuccess{ENodeStatus::SUCCESS, "semi_fail"})), result);
}

}

} // namespace NInfra::NPodAgent::NTestStaticSwitchNode
