#include "timed_cache_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/mock_node.h>
#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTimedCacheNodeTest {

Y_UNIT_TEST_SUITE(TimedCacheNodeSuite) {

static TLogger logger({});

TTimedCacheNodePtr CreateNode(TTickResult tickResult, const TDuration& timeout) {
    TTimedCacheNodePtr node = new TTimedCacheNode({1, "title"},  timeout);
    node->SetChildren({new TMockNode({1, "title"}, tickResult)});
    return node;
}

Y_UNIT_TEST(TestWithTwoChild) {
    const TTickResult tickResult = TNodeSuccess();
    auto node = CreateNode(tickResult, TDuration::Zero());
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        node->SetChildren(
            {
                new TMockNode({1, "title"}, tickResult)
                , new TMockNode({1, "title"}, tickResult)
            }
        )
        , yexception
        , "TimedCacheNode should have one child"
    );
}

Y_UNIT_TEST(TestWithRunningChild) {
    const size_t iter = 10;
    const TTickResult expected = TNodeSuccess(ENodeStatus::RUNNING, "running");
    const TDuration timeout = TDuration::Minutes(10);

    auto node = CreateNode(expected, timeout);

    for (size_t i = 0; i < iter; ++i) {
        auto result = node->Tick(MockTickContext(logger));
        UNIT_ASSERT_EQUAL(result, expected);
    }
}

Y_UNIT_TEST(TestWithErrorChild) {
    const size_t iter = 10;
    const TTickResult expected = TNodeError{"Error message"};
    const TDuration timeout = TDuration::Minutes(10);

    auto node = CreateNode(expected, timeout);

    for (size_t i = 0; i < iter; ++i) {
        auto result = node->Tick(MockTickContext(logger));
        UNIT_ASSERT_EQUAL(result, expected);
    }
}

Y_UNIT_TEST(TestWithTimeout) {
    const size_t iter = 10;
    const TDuration timeout = TDuration::Minutes(10);
    const TVector<TTickResult> results = {
        TNodeSuccess(ENodeStatus::SUCCESS, "success")
        , TNodeSuccess(ENodeStatus::FAILURE, "failure")
        , TNodeSuccess(ENodeStatus::IDLE, "idle")
    };

    for (const auto& result : results) {
        auto node = CreateNode(result, timeout);

        auto firstResult = node->Tick(MockTickContext(logger));
        UNIT_ASSERT_EQUAL(firstResult, result);

        for (size_t i = 0; i < iter; ++i) {
            auto currentResult = node->Tick(MockTickContext(logger));
            UNIT_ASSERT_C(currentResult, currentResult.Error().Message);
            UNIT_ASSERT_EQUAL_C(currentResult.Success().Status, result.Success().Status, ToString(currentResult.Success().Status));

            UNIT_ASSERT_STRING_CONTAINS(currentResult.Success().Message, "Skiped by timeout");
            UNIT_ASSERT_STRING_CONTAINS(currentResult.Success().Message, result.Success().Message);
        }
    }
}

Y_UNIT_TEST(TestTimeoutExpired) {
    const TDuration timeout = TDuration::MilliSeconds(500);
    const TVector<TTickResult> results = {
        TNodeSuccess(ENodeStatus::SUCCESS, "success")
        , TNodeSuccess(ENodeStatus::FAILURE, "failure")
        , TNodeSuccess(ENodeStatus::IDLE, "idle")
    };

    for (const auto& result : results) {
        auto node = CreateNode(result, timeout);

        {
            auto firstResult = node->Tick(MockTickContext(logger));
            UNIT_ASSERT_EQUAL(firstResult, result);
        }

        {
            // test before timeout
            auto currentResult = node->Tick(MockTickContext(logger));
            UNIT_ASSERT_C(currentResult, currentResult.Error().Message);
            UNIT_ASSERT_EQUAL_C(currentResult.Success().Status, result.Success().Status, ToString(currentResult.Success().Status));

            UNIT_ASSERT_STRING_CONTAINS(currentResult.Success().Message, "Skiped by timeout");
            UNIT_ASSERT_STRING_CONTAINS(currentResult.Success().Message, result.Success().Message);
        }

        {
            // test after timeout
            Sleep(TDuration(timeout));
            auto afterTimeoutResult = node->Tick(MockTickContext(logger));
            UNIT_ASSERT_EQUAL(afterTimeoutResult, result);
        }
    }
}

}

} // namespace NInfra::NPodAgent::NTimedCacheNodeTest
