#include "try_lock_node.h"

#include <util/string/builder.h>
#include <util/string/cast.h>

namespace NInfra::NPodAgent {

TLightRWLock TTryLockNode::LocksGlobalLock_;
THashMap<TString, TAtomicSharedPtr<TMutex>> TTryLockNode::Locks_;
THashMap<TString, TString> TTryLockNode::LockOwners_;

ENodeType TTryLockNode::GetType() const {
    return TTryLockNode::NODE_TYPE;
}

void TTryLockNode::SetChildren(TVector<TBasicTreeNodePtr>&& children) {
    Y_ENSURE(children.size() == 1, "TryLockNode should have one child");
    TBasicCompositeNode::SetChildren(std::move(children));
}

TAtomicSharedPtr<TMutex> TTryLockNode::GetLockById(const TString& lockId)  {
    {
        TReadGuardBase<TLightRWLock> guard(LocksGlobalLock_);
        auto ptr = Locks_.FindPtr(lockId);
        if (ptr) {
            return *ptr;
        }
    }

    {
        TWriteGuardBase<TLightRWLock> guard(LocksGlobalLock_);
        auto ptr = Locks_.FindPtr(lockId);
        if (ptr) {
            return *ptr;
        }

        Locks_[lockId] = MakeAtomicShared<TMutex>();
        LockOwners_[lockId] = "";
        return Locks_.at(lockId);
    }
}

void TTryLockNode::SetLockOwner(const TString lockId, const TString& ownerId) {
    TReadGuardBase<TLightRWLock> guard(LocksGlobalLock_);
    LockOwners_.at(lockId) = ownerId;
}

TString TTryLockNode::GetLockOwner(const TString& lockId) {
    TReadGuardBase<TLightRWLock> guard(LocksGlobalLock_);
    return LockOwners_.at(lockId);
}

TExpected<void, TString> TTryLockNode::TryLock() {
    if (IAmLockOwner_) {
        return TExpected<void, TString>::DefaultSuccess();
    }

    if (TTryGuard<TMutex> guard(*Lock_); guard.WasAcquired()) {
        const TString lockOwner = GetLockOwner(LockId_);
        if (lockOwner.empty()) {
            IAmLockOwner_ = true;
            SetLockOwner(LockId_, OwnerId_);
            return TExpected<void, TString>::DefaultSuccess();
        } else {
            return TExpected<void, TString>(
                TStringBuilder() << "Lock already aquired by '" << lockOwner << "'"
            );
        }
    } else {
        return TExpected<void, TString>(
            "Failed to aquire lock mutex"
        );
    }
}

void TTryLockNode::ReleaseLock() {
    // Not TTryGuard because we must release lock here
    TGuard<TMutex> guard(*Lock_);
    SetLockOwner(LockId_, "");
    IAmLockOwner_ = false;
}

TTickResult TTryLockNode::TickImpl(TTickContextPtr tickContext) {
    if (GetChildren().size() != 1) {
        return TNodeError{
            TStringBuilder() << "TryLockNode should have one child, but has "
                             << GetChildren().size()
        };
    }

    if (auto tryLockResult = TryLock(); !tryLockResult) {
        // TODO(chegoryu) Log tryLockResult.Error() somehow

        return TNodeSuccess(
            ENodeStatus::SUCCESS
            , ToString(ETryLockResult::TRY_LOCK_FAIL)
        );
    } else {
        auto child = GetChildren()[0];

        auto result = child->Tick(tickContext);
        // If child running just return result. Do not release lock.
        if (result && result.Success().Status == ENodeStatus::RUNNING) {
            return result;
        }

        // Release lock on all other results.
        ReleaseLock();

        // Return error as is
        if (!result) {
            return result;
        }

        if (result.Success().Status == ENodeStatus::SUCCESS) {
            return TNodeSuccess(ENodeStatus::SUCCESS, ToString(ETryLockResult::CHILD_SUCCESS));
        }

        if (result.Success().Status == ENodeStatus::FAILURE) {
            return TNodeSuccess(ENodeStatus::SUCCESS, ToString(ETryLockResult::CHILD_FAILURE));
        }

        return TNodeError{
            TStringBuilder() << "Unexpected TTryLockNode child result '" << ToString(result.Success().Status) << "'"
        };
    }
}

} // namespace NInfra::NPodAgent
