#pragma once

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/basic_composite_node.h>

#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

namespace NInfra::NPodAgent {

/**
 * TryLockNode
 *
 * Has exactly one child
 * It is a switch condition note: Never returns FAILURE, only SUCCESS + ETryLockResult, RUNNING, ERROR
 *
 * Try to acquire lock
 * On success does not release it until child returns non RUNNING result (node always has only one child)
 * Then returns CHILD_SUCCESS or CHILD_FAILURE
 * Otherwise returns TRY_LOCK_FAIL
 *
 */

class TTryLockNode;
using TTryLockNodePtr = TSimpleSharedPtr<TTryLockNode>;

class TTryLockNode : public TBasicCompositeNode {
public:
    enum ETryLockResult {
        TRY_LOCK_FAIL      /* "try_lock_fail" */,
        CHILD_SUCCESS      /* "child_success" */,
        CHILD_FAILURE      /* "child_failure" */,
    };

public:
    TTryLockNode(
        const TBasicTreeNodeDescriptor& descriptor
        , const TString& ownerId
        , const TString& lockId
    )
        : TBasicCompositeNode(descriptor, {})
        , OwnerId_(ownerId)
        , LockId_(lockId)
        , Lock_(GetLockById(lockId))
        , IAmLockOwner_(false)
    {
        Y_ENSURE(!OwnerId_.empty(), "Owner id can't be empty");
    }

    ~TTryLockNode() {
        if (IAmLockOwner_) {
            ReleaseLock();
        }
    }

    virtual ENodeType GetType() const override final;

    virtual void SetChildren(TVector<TBasicTreeNodePtr>&& children) override final;

private:
    virtual TTickResult TickImpl(TTickContextPtr tickContext) override final;

    static TAtomicSharedPtr<TMutex> GetLockById(const TString& lockId);
    static void SetLockOwner(const TString lockId, const TString& ownerId);
    static TString GetLockOwner(const TString& lockId);

    TExpected<void, TString> TryLock();
    void ReleaseLock();

public:
    static constexpr const ENodeType NODE_TYPE = ENodeType::TRY_LOCK;

private:
    static TLightRWLock LocksGlobalLock_;
    static THashMap<TString, TAtomicSharedPtr<TMutex>> Locks_;
    static THashMap<TString, TString> LockOwners_;

    const TString OwnerId_;
    const TString LockId_;
    const TAtomicSharedPtr<TMutex> Lock_;
    bool IAmLockOwner_;
};

} // namespace NInfra::NPodAgent
