#include "try_lock_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/mock_node.h>
#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/random/mersenne.h>

namespace NInfra::NPodAgent::NTryLockNodeTest {

Y_UNIT_TEST_SUITE(TryLockNodeSuite) {

static TLogger logger({});

// Stable unique id for every test
TString GenerateLockId() {
    static TMersenne<ui64> rnd(12345);
    return ToString(rnd.GenRand());
}

TTryLockNodePtr CreateNode(
    TTickResult tickResult
    , const TString& ownerId
    , const TString& lockId
) {
    TTryLockNodePtr node = new TTryLockNode({1, "title"}, ownerId, lockId);
    node->SetChildren({new TMockNode({1, "title"}, tickResult)});
    return node;
}

TTickResult GetLockCheckerResult(const TString& lockId) {
    auto node = CreateNode(TNodeSuccess(ENodeStatus::SUCCESS, "check"), "checker", lockId);
    return node->Tick(MockTickContext(logger));
}

void CheckLockAcquire(const TString& lockId) {
    auto result = GetLockCheckerResult(lockId);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL_C(result.Success().Status, ENodeStatus::SUCCESS, ToString(result.Success().Status));
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, ToString(TTryLockNode::ETryLockResult::TRY_LOCK_FAIL));
}

void CheckLockFree(const TString& lockId) {
    auto result = GetLockCheckerResult(lockId);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL_C(result.Success().Status, ENodeStatus::SUCCESS, ToString(result.Success().Status));
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, ToString(TTryLockNode::ETryLockResult::CHILD_SUCCESS));
}

Y_UNIT_TEST(TestWithEmptyOwnerId) {
    const TTickResult tickResult = TNodeSuccess();

    UNIT_ASSERT_EXCEPTION_CONTAINS(
        CreateNode(tickResult, "", "")
        , yexception
        , "Owner id can't be empty"
    );
}

Y_UNIT_TEST(TestWithTwoChild) {
    const TTickResult tickResult = TNodeSuccess();
    const TString ownerId = "owner";
    const TString lockId = GenerateLockId();

    auto node = CreateNode(tickResult, ownerId, lockId);
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        node->SetChildren(
            {
                new TMockNode({1, "title"}, tickResult)
                , new TMockNode({1, "title"}, tickResult)
            }
        )
        , yexception
        , "TryLockNode should have one child"
    );
}

Y_UNIT_TEST(TestWithRunningChild) {
    const size_t iter = 10;
    const TTickResult running = TNodeSuccess(ENodeStatus::RUNNING, "running");
    const TTickResult success = TNodeSuccess(ENodeStatus::SUCCESS, "success");
    const TTickResult expectedRunning = running;
    const TTickResult expectedSuccess = TNodeSuccess(ENodeStatus::SUCCESS, ToString(TTryLockNode::ETryLockResult::CHILD_SUCCESS));
    const TString ownerId = "owner";
    const TString lockId = GenerateLockId();

    auto node = CreateNode(running, ownerId, lockId);

    for (size_t i = 0; i < iter; ++i) {
        auto result = node->Tick(MockTickContext(logger));
        UNIT_ASSERT_EQUAL(result, expectedRunning);
        CheckLockAcquire(lockId);
    }

    node->SetChildren({new TMockNode({1, "title"}, success)});
    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_EQUAL(result, expectedSuccess);
    CheckLockFree(lockId);
}

Y_UNIT_TEST(TestWithSuccessChild) {
    const size_t iter = 10;
    const TTickResult success = TNodeSuccess(ENodeStatus::SUCCESS, "success");
    const TTickResult expectedSuccess = TNodeSuccess(ENodeStatus::SUCCESS, ToString(TTryLockNode::ETryLockResult::CHILD_SUCCESS));
    const TString ownerId = "owner";
    const TString lockId = GenerateLockId();

    auto node = CreateNode(success, ownerId, lockId);

    for (size_t i = 0; i < iter; ++i) {
        auto result = node->Tick(MockTickContext(logger));
        UNIT_ASSERT_EQUAL(result, expectedSuccess);
        CheckLockFree(lockId);
    }
}

Y_UNIT_TEST(TestWithFailureChild) {
    const size_t iter = 10;
    const TTickResult failure = TNodeSuccess(ENodeStatus::FAILURE, "failure");
    const TTickResult expectedFailure = TNodeSuccess(ENodeStatus::SUCCESS, ToString(TTryLockNode::ETryLockResult::CHILD_FAILURE));
    const TString ownerId = "owner";
    const TString lockId = GenerateLockId();

    auto node = CreateNode(failure, ownerId, lockId);

    for (size_t i = 0; i < iter; ++i) {
        auto result = node->Tick(MockTickContext(logger));
        UNIT_ASSERT_EQUAL(result, expectedFailure);
        CheckLockFree(lockId);
    }
}

Y_UNIT_TEST(TestWithErrorChild) {
    const size_t iter = 10;
    const TTickResult error = TNodeError{"Error message"};
    const TTickResult expectedError = error;
    const TString ownerId = "owner";
    const TString lockId = GenerateLockId();

    auto node = CreateNode(error, ownerId, lockId);

    for (size_t i = 0; i < iter; ++i) {
        auto result = node->Tick(MockTickContext(logger));
        UNIT_ASSERT_EQUAL(result, expectedError);
        CheckLockFree(lockId);
    }
}

Y_UNIT_TEST(TestReleaseLockFromAnotherThread) {
    const size_t iter = 10;
    const size_t cntThreads = 3;
    const TTickResult running = TNodeSuccess(ENodeStatus::RUNNING, "running");
    const TTickResult success = TNodeSuccess(ENodeStatus::SUCCESS, "success");
    const TTickResult expectedRunning = running;
    const TTickResult expectedSuccess = TNodeSuccess(ENodeStatus::SUCCESS, ToString(TTryLockNode::ETryLockResult::CHILD_SUCCESS));
    const TString ownerId = "owner";
    const TString lockId = GenerateLockId();

    // This test is quite specific
    // It is totaly OK on 4.4.171-70.1 with 16.04.6 LTS (Xenial Xerus) with wrong lock logic
    // However it is failed on 4.19.100-23 with 18.04.4 LTS (Bionic Beaver)
    // These assert are added here so that the person changing it thinks several times
    UNIT_ASSERT(cntThreads > 2);
    UNIT_ASSERT(iter >= 4);

    auto node = CreateNode(running, ownerId, lockId);
    TVector<TThreadPool> mtpQueue(cntThreads);
    for (size_t i = 0; i < cntThreads; ++i) {
        mtpQueue[i].Start(1);
    }

    for (size_t i = 0; i < iter; ++i) {
        if (i % 2 == 0) {
            node->SetChildren({new TMockNode({1, "title"}, running)});
            NThreading::TFuture<TTickResult> ft = NThreading::Async(
                [&]() -> TTickResult {
                    return node->Tick(MockTickContext(logger));
                }
                , mtpQueue[i % cntThreads]
            );

            auto result = ft.GetValueSync();
            UNIT_ASSERT_EQUAL(result, expectedRunning);
            CheckLockAcquire(lockId);
        } else {
            node->SetChildren({new TMockNode({1, "title"}, success)});
            NThreading::TFuture<TTickResult> ft = NThreading::Async(
                [&]() -> TTickResult {
                    return node->Tick(MockTickContext(logger));
                }
                , mtpQueue[i % cntThreads]
            );

            auto result = ft.GetValueSync();
            UNIT_ASSERT_EQUAL(result, expectedSuccess);
            CheckLockFree(lockId);
        }
    }

    for (size_t i = 0; i < cntThreads; ++i) {
        mtpQueue[i].Stop();
    }
}

Y_UNIT_TEST(TestReleaseLockFromDestructor) {
    const TTickResult running = TNodeSuccess(ENodeStatus::RUNNING, "running");
    const TTickResult expectedRunning = running;
    const TString ownerId = "owner";
    const TString lockId = GenerateLockId();

    {
        auto node = CreateNode(running, ownerId, lockId);
        auto result = node->Tick(MockTickContext(logger));
        UNIT_ASSERT_EQUAL(result, expectedRunning);
        CheckLockAcquire(lockId);
    }

    CheckLockFree(lockId);
}

}

} // namespace NInfra::NPodAgent::NTryLockNodeTest
