#include "capture_container_status_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/common.h>

#include <infra/pod_agent/libs/common/common.h>

#include <infra/pod_agent/libs/system_logs_sender/system_logs_session_parameters.h>

namespace NInfra::NPodAgent {

ENodeType TCaptureContainerStatusNode::GetType() const {
    return TCaptureContainerStatusNode::NODE_TYPE;
}

TExpected<NThreading::TFuture<TExpected<void, TSystemLogsSenderError>>, TTickResult> TCaptureContainerStatusNode::CaptureStatus(TTickContextPtr context) {
    auto feedback = ContainerStatusRepository_->CaptureContainerStatus(Container_);
    
    NLogEvent::TContainerAttemptFeedback event = CreateLogEvent(
        feedback
        , ContainerStatusRepository_->GetObjectIdsByHash(Container_.ObjectIdOrHash_)
        , ContainerStatusRepository_->GetCacheObjectIdsAndRevisionsByHash(Container_.ObjectIdOrHash_)
    );

    context.Get()->LogFrame.Get()->LogEvent(event);

    return SystemLogsSender_->SendMessage(Container_.ObjectIdOrHash_, SystemLogMessage(event));
}

NLogEvent::TContainerAttemptFeedback TCaptureContainerStatusNode::CreateLogEvent(
    const API::TContainerStatus::TAttemptFeedback& feedback
    , const TVector<TString>& objectIds
    , const TVector<TStatusRepositoryCommon::TCacheObject>& cacheObjects
) {

    NLogEvent::TContainerAttemptFeedback event;
    ui64 startTimeInMicroSeconds = TimestampToMicros(feedback.start_time());
    ui64 deathTimeInMicroSeconds = TimestampToMicros(feedback.death_time());

    for (const auto& id : objectIds) {
        *(event.add_ids()) = id;
    }

    for (const auto& cacheObject : cacheObjects) {
        *(event.add_cache_objects()) = NLogEvent::TPodAgentCacheObject(
            cacheObject.Id_
            , cacheObject.Revision_
        );
    }

    event.set_object_id_or_hash(Container_.ObjectIdOrHash_);
    event.set_object_type(GetObjectTypeFromEnum(Container_.ObjectType_));
    event.set_init_num(Container_.InitNum_);
    event.set_container_type(GetContainerTypeFromEnum(Container_.ContainerType_));
    event.set_return_code(feedback.return_code());
    event.set_stderr_tail(feedback.stderr_tail());
    event.set_stdout_tail(feedback.stdout_tail());
    event.set_fail_reason(feedback.fail_reason());
    event.set_state((NLogEvent::TContainerAttemptFeedback::EContainerState)feedback.state());
    event.set_start_time(startTimeInMicroSeconds);
    event.set_death_time(deathTimeInMicroSeconds);
    
    return event;
}

TString TCaptureContainerStatusNode::SystemLogMessage(NLogEvent::TContainerAttemptFeedback logEvent) {

    logEvent.set_stderr_tail("");
    logEvent.set_stdout_tail("");

    NLogEvent::TContainerHookSystemLogsEvent systemLogsEvent;
    systemLogsEvent.SetloggerName(TSystemLogsSessionParameters::DEPLOY_LOGGER_NAME_VALUE);
    systemLogsEvent.set_message(LogEventToMessage<NLogEvent::TContainerAttemptFeedback>(logEvent));
    
    if (Container_.ObjectType_ == NStatusRepositoryTypes::EObjectType::BOX) {
        systemLogsEvent.set_box(Container_.ObjectIdOrHash_);
    } else if (Container_.ObjectType_ == NStatusRepositoryTypes::EObjectType::WORKLOAD) {
        systemLogsEvent.set_workload(Container_.ObjectIdOrHash_);
    }

    systemLogsEvent.set_state(logEvent.state());
    systemLogsEvent.set_return_code(logEvent.return_code());
    systemLogsEvent.set_fail_reason(logEvent.fail_reason());
    systemLogsEvent.set_container_type(logEvent.container_type());
    systemLogsEvent.set_init_num(logEvent.init_num());
    
    return LogEventToMessage<NLogEvent::TContainerHookSystemLogsEvent>(systemLogsEvent);
}

TTickResult TCaptureContainerStatusNode::ProcessResult(TTickContextPtr/* context*/, TExpected<void, TSystemLogsSenderError>& result) {
    if (!bool(result)) {
        return TNodeError({ToString(result.Error())});
    }
 
    return TNodeSuccess(ENodeStatus::SUCCESS);
}

} // namespace NInfra::NPodAgent
