#include "capture_container_status_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>

#include <infra/pod_agent/libs/system_logs_sender/mock_system_logs_session.h>
#include <infra/pod_agent/libs/system_logs_sender/sidecars_system_logs_filter_impl.h>
#include <infra/pod_agent/libs/system_logs_sender/system_logs_sender_impl.h>
#include <infra/pod_agent/libs/system_logs_sender/system_logs_session_collection_impl.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestCaptureContainerStatusNode {

static TLogger logger({});

class TTestSystemLogsSession: public TMockSystemLogsSession
{
public:
    virtual ~TTestSystemLogsSession() = default;

    void SendMessage(const TString& message) override {
        SentMessage_ = message;
        SendMessageCalls_++;
    }

    TString SentMessage_ = "";
    int SendMessageCalls_ = 0;
};

class TCaptureWorkloadContainerBaseTestCase: public NUnitTest::TBaseTestCase {
public:
    TCaptureWorkloadContainerBaseTestCase()
        : Holder_(new TWorkloadStatusRepository())
        , SystemLogsSender_(new TSystemLogsSender(new TSystemLogsSessionCollection, new TFakeThreadPool, new TSystemLogsStatistics, new TSidecarsSystemLogsFilterImpl))
        , WorkloadId_("my_workload")
    {
        Holder_->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(WorkloadId_));
    }

    void TestCaptureContainerStatus() {
        TSystemLogsSessionPtr systemLogsSession = new TTestSystemLogsSession;

        SystemLogsSender_->Add(WorkloadId_, systemLogsSession);
        auto container = GetContainer();

        Holder_->UpdateContainerState(container, API::EContainerState_EXITED);
        Holder_->UpdateContainerFailReason(container, "fail_reason");
        Holder_->UpdateContainerStderr(container, "stderr_text");
        Holder_->UpdateContainerStdout(container, "stdout_text");
        Holder_->UpdateContainerReturnCode(container, 123);

        TCaptureContainerStatusNodePtr node = CreateNode();
        auto result = node->Tick(MockTickContext(logger));
        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(TNodeSuccess(ENodeStatus::SUCCESS), result.Success());
        UNIT_ASSERT_EQUAL(((TTestSystemLogsSession*)systemLogsSession.Get())->SendMessageCalls_, 1);

        auto status = Holder_->GetObjectStatus(WorkloadId_);
        UNIT_ASSERT_EQUAL(GetContainerStatus(status).last().fail_reason(), "fail_reason");
        UNIT_ASSERT_EQUAL(GetContainerStatus(status).last().state(), API::EContainerState_EXITED);
        UNIT_ASSERT_EQUAL(GetContainerStatus(status).last().stderr_tail(), "stderr_text");
        UNIT_ASSERT_EQUAL(GetContainerStatus(status).last().stdout_tail(), "stdout_text");
        UNIT_ASSERT_EQUAL(GetContainerStatus(status).last().return_code(), 123);

        UNIT_ASSERT_EQUAL(GetContainerStatus(status).current().fail_reason(), "");
        UNIT_ASSERT_EQUAL(GetContainerStatus(status).current().state(), API::EContainerState_UNKNOWN);
        UNIT_ASSERT_EQUAL(GetContainerStatus(status).current().stderr_tail(), "");
        UNIT_ASSERT_EQUAL(GetContainerStatus(status).current().stdout_tail(), "");
        UNIT_ASSERT_EQUAL(GetContainerStatus(status).current().return_code(), 0);
        
        auto logEvent = node->CreateLogEvent(GetContainerStatus(status).last(), {}, {});
        auto systemLogsMessage = node->SystemLogMessage(logEvent);
        UNIT_ASSERT_EQUAL(((TTestSystemLogsSession*)systemLogsSession.Get())->SentMessage_, systemLogsMessage);

        SystemLogsSender_->RemoveAll();
        node = CreateNode();
        result = node->Tick(MockTickContext(logger));
        UNIT_ASSERT(!(bool)result);
    }

private:
    virtual NStatusRepositoryTypes::TContainerDescription::EContainerType GetContainerType() const = 0;
    virtual API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const = 0;

    NStatusRepositoryTypes::TContainerDescription GetContainer() const {
        return NStatusRepositoryTypes::TContainerDescription(
            WorkloadId_
            , NStatusRepositoryTypes::EObjectType::WORKLOAD
            , GetContainerType()
        );
    }

    TCaptureContainerStatusNodePtr CreateNode() {
        return new TCaptureContainerStatusNode(
            TBasicTreeNodeDescriptor{1, "title"}
            , Holder_
            , SystemLogsSender_
            , GetContainer()
        );
    }

public:
    TWorkloadStatusRepositoryPtr Holder_;
    ISystemLogsSenderPtr SystemLogsSender_;
    const TString WorkloadId_;
};

class TCaptureWorkloadContainerReadinessTestCase: public TCaptureWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.readiness_status().container_status();
    }
};

class TCaptureWorkloadContainerLivenessTestCase: public TCaptureWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::LIVENESS;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.liveness_status().container_status();
    }
};

class TCaptureWorkloadContainerStartTestCase: public TCaptureWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::START;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.start();
    }
};

class TCaptureWorkloadContainerStopTestCase: public TCaptureWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.stop_status().container_status();
    }
};

class TCaptureWorkloadContainerDestroyTestCase: public TCaptureWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::DESTROY;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.destroy_status().container_status();
    }
};

Y_UNIT_TEST_SUITE_F(CaptureWorkloadContainerReadinessStatusNodeSuite, TCaptureWorkloadContainerReadinessTestCase) {

Y_UNIT_TEST(TestCaptureContainerStatus) {
    TestCaptureContainerStatus();
}

}

Y_UNIT_TEST_SUITE_F(CaptureWorkloadContainerLivenessStatusNodeSuite, TCaptureWorkloadContainerLivenessTestCase) {

Y_UNIT_TEST(TestCaptureContainerStatus) {
    TestCaptureContainerStatus();
}

}

Y_UNIT_TEST_SUITE_F(CaptureWorkloadContainerStartStatusNodeSuite, TCaptureWorkloadContainerStartTestCase) {

Y_UNIT_TEST(TestCaptureContainerStatus) {
    TestCaptureContainerStatus();
}

}

Y_UNIT_TEST_SUITE_F(CaptureWorkloadContainerStopStatusNodeSuite, TCaptureWorkloadContainerStopTestCase) {

Y_UNIT_TEST(TestCaptureContainerStatus) {
    TestCaptureContainerStatus();
}

}

Y_UNIT_TEST_SUITE_F(CaptureWorkloadContainerDestroyStatusNodeSuite, TCaptureWorkloadContainerDestroyTestCase) {

Y_UNIT_TEST(TestCaptureContainerStatus) {
    TestCaptureContainerStatus();
}

}

} // namespace NInfra::NPodAgent::NTestCaptureContainerStatusNode
