#include "check_container_retries_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestCheckContainerStopRetriesNode  {

static TLogger logger({});

TCheckContainerRetriesNodePtr GetCheckContainerRetriesNode(
    TWorkloadStatusRepositoryPtr statusRepository
    , NStatusRepositoryTypes::TContainerDescription::EContainerType containerType
    , const TString& workloadId
    , ui32 maxTries
) {
    return new TCheckContainerRetriesNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , statusRepository
        , NStatusRepositoryTypes::TContainerDescription(
            workloadId
            , NStatusRepositoryTypes::EObjectType::WORKLOAD
            , containerType
        )
        , maxTries
    );
}

class ITestCheckContainerRetriesNodeCanon {
public:
    ITestCheckContainerRetriesNodeCanon()
        : WorkloadStatusRepository_(new TWorkloadStatusRepository())
    {}

    void DoTest() {
        Test();
    }

protected:
    virtual void Test() = 0;

protected:
    TWorkloadStatusRepositoryPtr WorkloadStatusRepository_;

};

Y_UNIT_TEST_SUITE(CheckContainerRetriesNodeSuite) {

Y_UNIT_TEST(TestSuccess) {
    class TTest : public ITestCheckContainerRetriesNodeCanon {
    protected:
        void Test() override {
            const TString id = "my_workload";
            WorkloadStatusRepository_->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

            TCheckContainerRetriesNodePtr node = GetCheckContainerRetriesNode(
                WorkloadStatusRepository_
                , NStatusRepositoryTypes::TContainerDescription::EContainerType::DESTROY
                , id
                , 1
            );
            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestFailure) {
    class TTest : public ITestCheckContainerRetriesNodeCanon {
    protected:
        void Test() override {
            const TString id = "my_workload";
            WorkloadStatusRepository_->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

            NStatusRepositoryTypes::TContainerDescription container(
                id
                , NStatusRepositoryTypes::EObjectType::WORKLOAD
                , NStatusRepositoryTypes::TContainerDescription::EContainerType::DESTROY
            );

            WorkloadStatusRepository_->IncrementContainerZeroReturnCodeCounter(container);
            WorkloadStatusRepository_->IncrementContainerPositiveReturnCodeCounter(container);
            WorkloadStatusRepository_->IncrementContainerOomCounter(container);
            WorkloadStatusRepository_->IncrementContainerKilledExternallyCounter(container);
            WorkloadStatusRepository_->IncrementContainerSystemFailureCounter(container);

            TCheckContainerRetriesNodePtr node = GetCheckContainerRetriesNode(
                WorkloadStatusRepository_
                , NStatusRepositoryTypes::TContainerDescription::EContainerType::DESTROY
                , id
                , 5
            );
            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
            UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "Workload 'my_workload' exceeded destroy tries limit: 5");
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadStatusRepositoryNotDefined) {
    class TTest : public ITestCheckContainerRetriesNodeCanon {
    protected:
        void Test() override {
            const TString id = "my_workload";

            UNIT_ASSERT_EXCEPTION(
                GetCheckContainerRetriesNode(
                    nullptr
                    , NStatusRepositoryTypes::TContainerDescription::EContainerType::DESTROY
                    , id
                    , 2
                )
                , yexception
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestMaxRetriesIsZero) {
    class TTest : public ITestCheckContainerRetriesNodeCanon {
    protected:
        void Test() override {
            const TString id = "my_workload";
            WorkloadStatusRepository_->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

            TCheckContainerRetriesNodePtr node = GetCheckContainerRetriesNode(
                WorkloadStatusRepository_
                , NStatusRepositoryTypes::TContainerDescription::EContainerType::DESTROY
                , id
                , 0
            );
            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "MaxTries is zero");

        }
    };

    TTest test;
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTestCheckContainerStopRetriesNode
