#include "container_start_time_expired_with_backoff_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestContainerStartTimeExpiredWithBackoffNode {

Y_UNIT_TEST_SUITE(PortoContainerStartTimeExpiredNodeWithBackoffSuite) {

static TLogger logger({});

TContainerStartTimeExpiredNodeWithBackOffPtr CreateNode(
    const TStatusRepositoryCommonPtr statusRepository
    , const NStatusRepositoryTypes::TContainerDescription& container
    , TDuration offset
    , TDuration scale
    , ui64 exponentBase
    , TDuration capTime
) {
    return TContainerStartTimeExpiredNodeWithBackOffPtr(new TContainerStartTimeExpiredNodeWithBackOff(
        TBasicTreeNodeDescriptor{1, "title"}
        , statusRepository
        , container
        , offset
        , scale
        , exponentBase
        , capTime
    ));
}

Y_UNIT_TEST(TestStartTimeExpired) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TString id = "my_workload";
    NStatusRepositoryTypes::TContainerDescription container = NStatusRepositoryTypes::TContainerDescription(
        id
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS
    );

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    holder->UpdateContainerStartTime(container, TInstant::Seconds(10));
    holder->CaptureContainerStatus(container);

    TDuration duration = TDuration::Seconds(60);
    auto node = CreateNode(holder, container, duration, TDuration::FromValue(0), 1, duration);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL_C(0, context->GetLongTickPeriodDuration(), context->GetLongTickPeriodDuration());
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
}

Y_UNIT_TEST(TestStartTimeNotExpired) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TString id = "my_workload";
    NStatusRepositoryTypes::TContainerDescription container = NStatusRepositoryTypes::TContainerDescription(
        id
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS
    );

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    holder->UpdateContainerStartTime(container, TInstant::Seconds(4000000000));
    holder->CaptureContainerStatus(container);

    TDuration duration = TDuration::Seconds(60);
    auto node = CreateNode(holder, container, duration, TDuration::FromValue(0), 1, duration);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
}

Y_UNIT_TEST(TestStartTimeNowNotExpired) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TString id = "my_workload";
    NStatusRepositoryTypes::TContainerDescription container = NStatusRepositoryTypes::TContainerDescription(
        id
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS
    );

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    holder->UpdateContainerStartTime(container, Now());
    holder->CaptureContainerStatus(container);

    TDuration duration = TDuration::Seconds(60);
    auto node = CreateNode(holder, container, duration, TDuration::FromValue(0), 1, duration);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT(labs(60000 - (long) context->GetLongTickPeriodDuration()) < 1000);
    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
}

Y_UNIT_TEST(TestStartTimeBackoff) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TString id = "my_workload";
    NStatusRepositoryTypes::TContainerDescription container = NStatusRepositoryTypes::TContainerDescription(
        id
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS
    );

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    holder->UpdateContainerStartTime(container, Now() - TDuration::Seconds(59));
    holder->UpdateContainerReturnCode(container, 0);
    holder->UpdateContainerConsecutiveFailuresAndSuccessesCounter(container, 0);
    holder->CaptureContainerStatus(container);

    TDuration second = TDuration::Seconds(1);
    auto node = CreateNode(holder, container, second, second, 1000, second * 1000);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    holder->UpdateContainerReturnCode(container, 0);
    holder->UpdateContainerConsecutiveFailuresAndSuccessesCounter(container, 0);

    context = MockTickContext(logger);
    result = node->Tick(context);

    UNIT_ASSERT(labs(1000000 - 59000 - (long) context->GetLongTickPeriodDuration()) < 1000);
    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
}

Y_UNIT_TEST(TestStartTimeNoBackoff) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TString id = "my_workload";
    NStatusRepositoryTypes::TContainerDescription container = NStatusRepositoryTypes::TContainerDescription(
        id
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS
    );

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    holder->UpdateContainerStartTime(container, Now() - TDuration::Seconds(59));
    holder->UpdateContainerReturnCode(container, -1);
    holder->UpdateContainerConsecutiveFailuresAndSuccessesCounter(container, -1);
    holder->CaptureContainerStatus(container);

    TDuration second = TDuration::Seconds(1);
    auto node = CreateNode(holder, container, second, second, 1000, second * 1000);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    context = MockTickContext(logger);
    result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
}

Y_UNIT_TEST(TestStartTimeDoubleCheck) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TString id = "my_workload";
    NStatusRepositoryTypes::TContainerDescription container = NStatusRepositoryTypes::TContainerDescription(
        id
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS
    );

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    holder->UpdateContainerStartTime(container, Now() - TDuration::Seconds(59));
    holder->UpdateContainerReturnCode(container, 0);
    holder->CaptureContainerStatus(container);

    TDuration second = TDuration::Seconds(1);
    auto node = CreateNode(holder, container, second, second, 1000, second * 1000);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    context = MockTickContext(logger);
    result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
}

Y_UNIT_TEST(TestStartTimeBackoffWorkloadStop) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TString id = "my_workload";
    NStatusRepositoryTypes::TContainerDescription container = NStatusRepositoryTypes::TContainerDescription(
        id
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP
    );

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    holder->UpdateContainerStartTime(container, Now() - TDuration::Seconds(59));
    holder->UpdateContainerReturnCode(container, 1);
    holder->UpdateContainerConsecutiveFailuresAndSuccessesCounter(container, 1);
    holder->CaptureContainerStatus(container);

    TDuration second = TDuration::Seconds(1);
    auto node = CreateNode(holder, container, second, second, 1000, second * 1000);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    holder->UpdateContainerReturnCode(container, 1);
    holder->UpdateContainerConsecutiveFailuresAndSuccessesCounter(container, 1);

    context = MockTickContext(logger);
    result = node->Tick(context);

    UNIT_ASSERT(labs(1000000 - 59000 - (long) context->GetLongTickPeriodDuration()) < 1000);
    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
}

Y_UNIT_TEST(TestStartTimeNoBackoffWorkloadStop) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TString id = "my_workload";
    NStatusRepositoryTypes::TContainerDescription container = NStatusRepositoryTypes::TContainerDescription(
        id
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP
    );

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    holder->UpdateContainerStartTime(container, Now() - TDuration::Seconds(59));
    holder->UpdateContainerReturnCode(container, 0);
    holder->UpdateContainerConsecutiveFailuresAndSuccessesCounter(container, 0);
    holder->CaptureContainerStatus(container);

    TDuration second = TDuration::Seconds(1);
    auto node = CreateNode(holder, container, second, second, 1000, second * 1000);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    context = MockTickContext(logger);
    result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
}

Y_UNIT_TEST(TestStartTimeDoubleCheckWorkloadStop) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TString id = "my_workload";
    NStatusRepositoryTypes::TContainerDescription container = NStatusRepositoryTypes::TContainerDescription(
        id
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP
    );

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    holder->UpdateContainerStartTime(container, Now() - TDuration::Seconds(59));
    holder->UpdateContainerReturnCode(container, 1);
    holder->CaptureContainerStatus(container);

    TDuration second = TDuration::Seconds(1);
    auto node = CreateNode(holder, container, second, second, 1000, second * 1000);
    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    context = MockTickContext(logger);
    result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
}

}

} // namespace NInfra::NPodAgent::NTestContainerStartTimeExpiredWithBackoffNode
