#include "feedback_container_death_time_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestFeedbackContainerDeathTimeNode {

static TLogger logger({});

class TFeedbackWorkloadContainerTestCase: public NUnitTest::TBaseTestCase {
public:
    TFeedbackWorkloadContainerTestCase()
        : Holder_(new TWorkloadStatusRepository())
        , WorkloadId_("my_workload")
        , ContainerName_("my_container")
    {
        Holder_->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(WorkloadId_));
    }

    void TestUpdateTime() {
        struct TMyPortoClient : public TMockPortoClient {
            TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
                Property = property;
                Name = name;
                return ToString(12345);
            }
            TPortoContainerName Name{""};
            EPortoContainerProperty Property;
        };

        TPortoClientPtr myPorto = new TMyPortoClient();
        TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

        auto node = CreateNode(porto);
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

        UNIT_ASSERT_EQUAL(12345, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).current().death_time().seconds());
        UNIT_ASSERT_EQUAL(0, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).current().death_time().nanos());

        UNIT_ASSERT_EQUAL(ContainerName_, ((TMyPortoClient*) myPorto.Get())->Name);
        UNIT_ASSERT_EQUAL(EPortoContainerProperty::DeathTimeRaw, ((TMyPortoClient*) myPorto.Get())->Property);
    }

    void TestFailingPorto() {
        struct TMyPortoClient : public TMockPortoClient {
            TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
                Property = property;
                Name = name;
                return TPortoError{EPortoError::Unknown, "GetProperty", ""};
            }
            TPortoContainerName Name{""};
            EPortoContainerProperty Property;
        };

        TPortoClientPtr myPorto = new TMyPortoClient();
        TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

        auto node = CreateNode(porto);
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(!result, result.Success().Message);
        UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "GetProperty");

        UNIT_ASSERT_EQUAL(ContainerName_, ((TMyPortoClient*) myPorto.Get())->Name);
        UNIT_ASSERT_EQUAL(EPortoContainerProperty::DeathTimeRaw, ((TMyPortoClient*) myPorto.Get())->Property);
    }

private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType GetContainerType() const {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const {
        return status.readiness_status().container_status();
    }

    TFeedbackContainerDeathTimeNodePtr CreateNode(const TAsyncPortoClientPtr porto) {
        return new TFeedbackContainerDeathTimeNode(
            {1, "title"}
            , Holder_
            , porto
            , NStatusRepositoryTypes::TContainerDescription(
                WorkloadId_
                , NStatusRepositoryTypes::EObjectType::WORKLOAD
                , GetContainerType()
            )
            , ContainerName_
        );
    }

public:
    TWorkloadStatusRepositoryPtr Holder_;
    const TString WorkloadId_;
    const TPortoContainerName ContainerName_;
};


Y_UNIT_TEST_SUITE_F(FeedbackWorkloadContainerDeathTimeNodeSuite, TFeedbackWorkloadContainerTestCase)  {

Y_UNIT_TEST(TestUpdateTime) {
    TestUpdateTime();
}

Y_UNIT_TEST(TestFailingPorto) {
    TestFailingPorto();
}
}

} // namespace NInfra::NPodAgent::NTestFeedbackContainerDeathTimeNode
