#include "feedback_container_state_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestFeedbackContainerStartNode {

static TLogger logger({});

class TFeedbackWorkloadContainerBaseTestCase: public NUnitTest::TBaseTestCase {
public:
    TFeedbackWorkloadContainerBaseTestCase()
        : Holder_(new TWorkloadStatusRepository())
        , WorkloadId_("my_workload")
    {
        Holder_->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(WorkloadId_));
    }

    void TestUpdateState() {
        API::EContainerState state = API::EContainerState_RUNNING;
        TFeedbackContainerStateNodePtr node = CreateNode(state);
        auto result = node->Tick(MockTickContext(logger));
        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(TNodeSuccess(ENodeStatus::SUCCESS), result.Success());
        API::TWorkloadStatus status = Holder_->GetObjectStatus(WorkloadId_);
        UNIT_ASSERT_EQUAL(state, GetContainerStatus(status).current().state());

        API::EContainerState timeoutState = API::EContainerState_TIMEOUT;
        node = CreateNode(timeoutState);
        result = node->Tick(MockTickContext(logger));
        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(result.Success(), TNodeSuccess(ENodeStatus::SUCCESS));
        status = Holder_->GetObjectStatus(WorkloadId_);
        UNIT_ASSERT_EQUAL(timeoutState, GetContainerStatus(status).current().state());
        UNIT_ASSERT_EQUAL_C(GetContainerStatus(status).timeout_counter(), 1, GetContainerStatus(status).timeout_counter());
    }

private:
    virtual NStatusRepositoryTypes::TContainerDescription::EContainerType GetContainerType() const = 0;
    virtual API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const = 0;

    TFeedbackContainerStateNodePtr CreateNode(API::EContainerState state) {
        return new TFeedbackContainerStateNode(
            TBasicTreeNodeDescriptor{1, "title"}
            , Holder_
            , NStatusRepositoryTypes::TContainerDescription(
                WorkloadId_
                , NStatusRepositoryTypes::EObjectType::WORKLOAD
                , GetContainerType()
            )
            , state
        );
    }

protected:
    TWorkloadStatusRepositoryPtr Holder_;
    const TString WorkloadId_;
};

class TFeedbackWorkloadContainerReadinessTestCase: public TFeedbackWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.readiness_status().container_status();
    }
};

class TFeedbackWorkloadContainerLivenessTestCase: public TFeedbackWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::LIVENESS;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.liveness_status().container_status();
    }
};

class TFeedbackWorkloadContainerStartTestCase: public TFeedbackWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::START;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.start();
    }
};

class TFeedbackWorkloadContainerStopTestCase: public TFeedbackWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.stop_status().container_status();
    }
};

Y_UNIT_TEST_SUITE_F(FeedbackWorkloadContainerReadinessStateNodeSuite, TFeedbackWorkloadContainerReadinessTestCase) {

Y_UNIT_TEST(TestReadinessUpdateState) {
    TestUpdateState();
}

}

Y_UNIT_TEST_SUITE_F(FeedbackWorkloadContainerLivenessStateNodeSuite, TFeedbackWorkloadContainerLivenessTestCase) {

Y_UNIT_TEST(TestLivenessUpdateState) {
    TestUpdateState();
}

}

Y_UNIT_TEST_SUITE_F(FeedbackWorkloadContainerStartStateNodeSuite, TFeedbackWorkloadContainerStartTestCase) {

Y_UNIT_TEST(TestReadinessUpdateState) {
    TestUpdateState();
}

}

Y_UNIT_TEST_SUITE_F(FeedbackWorkloadContainerStopStateNodeSuite, TFeedbackWorkloadContainerStopTestCase) {

Y_UNIT_TEST(TestReadinessUpdateState) {
    TestUpdateState();
}

}

} // namespace NInfra::NPodAgent::NTestFeedbackContainerStartNode
