#include "feedback_container_status_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestFeedbackContainerStatusNode {

static TLogger logger({});

class TFeedbackWorkloadContainerBaseTestCase: public NUnitTest::TBaseTestCase {
public:
    TFeedbackWorkloadContainerBaseTestCase()
        : Holder_(new TWorkloadStatusRepository())
        , WorkloadId_("my_workload")
    {
        Holder_->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(WorkloadId_));
    }

    void TestExitCodeOOM() {
        struct TMyPortoClient : public TMockPortoClient {
            TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& /*name*/, EPortoContainerProperty /*property*/, int /*flags*/) override {
                return TString("-99");
            }
        };

        TPortoClientPtr myPorto = new TMyPortoClient();
        TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

        auto node = CreateNode(porto);
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

        UNIT_ASSERT_EQUAL(API::EContainerState_OUT_OF_MEMORY, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).current().state());
        UNIT_ASSERT_EQUAL(GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).current().return_code(), -99);
        UNIT_ASSERT_EQUAL(1, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).oom_counter());
        UNIT_ASSERT_EQUAL(1, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).time_limit().consecutive_failures_counter());
        UNIT_ASSERT_EQUAL(0, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).time_limit().consecutive_successes_counter());
    }

    void TestExitCodeZero() {
        struct TMyPortoClient : public TMockPortoClient {
            TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& /*name*/, EPortoContainerProperty /*property*/, int /*flags*/) override {
                return TString("0");
            }
        };

        TPortoClientPtr myPorto = new TMyPortoClient();
        TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

        auto node = CreateNode(porto);
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

        UNIT_ASSERT_EQUAL(API::EContainerState_EXITED, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).current().state());
        UNIT_ASSERT_EQUAL(0, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).current().return_code());
        UNIT_ASSERT_EQUAL(1, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).zero_return_code_counter());
        UNIT_ASSERT_EQUAL(0, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).time_limit().consecutive_failures_counter());
        UNIT_ASSERT_EQUAL(1, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).time_limit().consecutive_successes_counter());
    }

    void TestExitCodePositive() {
        struct TMyPortoClient : public TMockPortoClient {
            TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& /*name*/, EPortoContainerProperty /*property*/, int /*flags*/) override {
                return TString("42");
            }
        };

        TPortoClientPtr myPorto = new TMyPortoClient();
        TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

        auto node = CreateNode(porto);
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

        UNIT_ASSERT_EQUAL(API::EContainerState_EXITED, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).current().state());
        UNIT_ASSERT(GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).current().return_code() > 0);
        UNIT_ASSERT_EQUAL(1, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).positive_return_code_counter());
        UNIT_ASSERT_EQUAL(1, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).time_limit().consecutive_failures_counter());
        UNIT_ASSERT_EQUAL(0, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).time_limit().consecutive_successes_counter());
    }

    void TestExitCodeKilled() {
        struct TMyPortoClient : public TMockPortoClient {
            TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& /*name*/, EPortoContainerProperty /*property*/, int /*flags*/) override {
                return TString("-42");
            }
        };

        TPortoClientPtr myPorto = new TMyPortoClient();
        TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

        auto node = CreateNode(porto);
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

        UNIT_ASSERT_EQUAL(API::EContainerState_KILLED_EXTERNALLY, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).current().state());
        UNIT_ASSERT(GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).current().return_code() < 0);
        UNIT_ASSERT_EQUAL(1, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).killed_externally_counter());
        UNIT_ASSERT_EQUAL(1, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).time_limit().consecutive_failures_counter());
        UNIT_ASSERT_EQUAL(0, GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).time_limit().consecutive_successes_counter());
    }

    void TestOnPortoError() {
        struct TMyPortoFailingClient : public TMockPortoClient {
            TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& /*name*/, EPortoContainerProperty /*property*/, int /*flags*/) override {
                return TPortoError{EPortoError::Unknown, TString("GetProperty"), TString("")};
            }
        };

        TPortoClientPtr myPorto = new TMyPortoFailingClient();
        TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

        auto node = CreateNode(porto);
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(!result, result.Success().Message);
        UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "GetProperty");
    }

private:
    virtual NStatusRepositoryTypes::TContainerDescription::EContainerType GetContainerType() const = 0;
    virtual API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const = 0;

    TFeedbackContainerStatusNodePtr CreateNode(const TAsyncPortoClientPtr porto) {
        return new TFeedbackContainerStatusNode(
            {1, "title"}
            , Holder_
            , porto
            , NStatusRepositoryTypes::TContainerDescription(
                WorkloadId_
                , NStatusRepositoryTypes::EObjectType::WORKLOAD
                , GetContainerType()
            )
            , ContainerName_
        );
    }

protected:
    TWorkloadStatusRepositoryPtr Holder_;
    const TString WorkloadId_;
    const TPortoContainerName ContainerName_{""};
};

class TFeedbackWorkloadContainerReadinessTestCase: public TFeedbackWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.readiness_status().container_status();
    }
};

class TFeedbackWorkloadContainerStartTestCase: public TFeedbackWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::START;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.start();
    }
};

class TFeedbackWorkloadContainerStopTestCase: public TFeedbackWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.stop_status().container_status();
    }
};


Y_UNIT_TEST_SUITE_F(FeedbackWorkloadContainerReadinessStatusNodeSuite, TFeedbackWorkloadContainerReadinessTestCase)  {

Y_UNIT_TEST(TestExitCodeOOM) {
    TestExitCodeOOM();
}

Y_UNIT_TEST(TestExitCodeZero) {
    TestExitCodeZero();
}

Y_UNIT_TEST(TestExitCodePositive) {
    TestExitCodePositive();
}

Y_UNIT_TEST(TestExitCodeKilled) {
    TestExitCodeKilled();
}

Y_UNIT_TEST(TestOnPortoError) {
    TestOnPortoError();
}
}

Y_UNIT_TEST_SUITE_F(FeedbackWorkloadContainerStartStatusNodeSuite, TFeedbackWorkloadContainerStartTestCase)  {

Y_UNIT_TEST(TestExitCodeOOM) {
    TestExitCodeOOM();
}

Y_UNIT_TEST(TestExitCodeZero) {
    TestExitCodeZero();
}

Y_UNIT_TEST(TestExitCodePositive) {
    TestExitCodePositive();
}

Y_UNIT_TEST(TestExitCodeKilled) {
    TestExitCodeKilled();
}

Y_UNIT_TEST(TestOnPortoError) {
    TestOnPortoError();
}
}

Y_UNIT_TEST_SUITE_F(FeedbackWorkloadContainerStopStatusNodeSuite, TFeedbackWorkloadContainerStopTestCase)  {

Y_UNIT_TEST(TestExitCodeOOM) {
    TestExitCodeOOM();
}

Y_UNIT_TEST(TestExitCodeZero) {
    TestExitCodeZero();
}

Y_UNIT_TEST(TestExitCodePositive) {
    TestExitCodePositive();
}

Y_UNIT_TEST(TestExitCodeKilled) {
    TestExitCodeKilled();
}

Y_UNIT_TEST(TestOnPortoError) {
    TestOnPortoError();
}
}

} // namespace NInfra::NPodAgent::NTestFeedbackContainerStatusNode
