#include "feedback_container_stdout_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestFeedbackContainerStdoutNode {

static TLogger logger({});

class TFeedbackWorkloadContainerBaseTestCase: public NUnitTest::TBaseTestCase {
public:
    TFeedbackWorkloadContainerBaseTestCase()
        : Holder_(new TWorkloadStatusRepository())
        , WorkloadId_("my_workload")
    {
        Holder_->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(WorkloadId_));
    }

    void TestStdout() {
        struct TMyPortoClient : public TMockPortoClient {
            TExpected<TString, TPortoError> GetStdout(const TPortoContainerName& name, int offset, int length, int flags) override {
                Name_ = (TString)name;
                Offset_ = offset;
                Length_ = length;
                Flags_ = flags;

                return TString("This is container's stdout");
            }

            TString Name_;
            int Offset_;
            int Length_;
            int Flags_;
        };

        TPortoClientPtr myPorto = new TMyPortoClient();
        TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

        auto node = CreateNode(porto);
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL_C(ENodeStatus::SUCCESS, result.Success().Status, ToString(result.Success().Status));

        TString stdout = GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).current().stdout_tail();
        UNIT_ASSERT_EQUAL_C(stdout, "This is container's stdout", stdout);
        UNIT_ASSERT_EQUAL_C(ContainerName_, ((TMyPortoClient*)myPorto.Get())->Name_, ((TMyPortoClient*)myPorto.Get())->Name_);
        UNIT_ASSERT_EQUAL_C(Offset_, ((TMyPortoClient*)myPorto.Get())->Offset_, ((TMyPortoClient*)myPorto.Get())->Offset_);
        UNIT_ASSERT_EQUAL_C(Length_, ((TMyPortoClient*)myPorto.Get())->Length_, ((TMyPortoClient*)myPorto.Get())->Length_);
        UNIT_ASSERT_EQUAL_C(Flags_, ((TMyPortoClient*)myPorto.Get())->Flags_, ((TMyPortoClient*)myPorto.Get())->Flags_);
    }

    void TestOnPortoError() {
        struct TMyPortoFailingClient : public TMockPortoClient {
            TExpected<TString, TPortoError> GetStdout(const TPortoContainerName& name, int offset, int length, int flags) override {
                Name_ = (TString)name;
                Offset_ = offset;
                Length_ = length;
                Flags_ = flags;

                return TPortoError{EPortoError::Unknown, TString("GetStdout"), TString("")};
            }

            TString Name_;
            int Offset_;
            int Length_;
            int Flags_;
        };

        TPortoClientPtr myPorto = new TMyPortoFailingClient();
        TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

        auto node = CreateNode(porto);
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(!result, result.Success().Message);
        UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "GetStdout");

        UNIT_ASSERT_EQUAL_C(ContainerName_, ((TMyPortoFailingClient*)myPorto.Get())->Name_, ((TMyPortoFailingClient*)myPorto.Get())->Name_);
        UNIT_ASSERT_EQUAL_C(Offset_, ((TMyPortoFailingClient*)myPorto.Get())->Offset_, ((TMyPortoFailingClient*)myPorto.Get())->Offset_);
        UNIT_ASSERT_EQUAL_C(Length_, ((TMyPortoFailingClient*)myPorto.Get())->Length_, ((TMyPortoFailingClient*)myPorto.Get())->Length_);
        UNIT_ASSERT_EQUAL_C(Flags_, ((TMyPortoFailingClient*)myPorto.Get())->Flags_, ((TMyPortoFailingClient*)myPorto.Get())->Flags_);
    }

    void TestOnInvalidDataPortoError() {
        struct TMyPortoFailingClient : public TMockPortoClient {
            TExpected<TString, TPortoError> GetStdout(const TPortoContainerName& name, int offset, int length, int flags) override {
                Name_ = (TString)name;
                Offset_ = offset;
                Length_ = length;
                Flags_ = flags;

                return TPortoError{EPortoError::InvalidData, TString(""), TString(""), TString("File is non-regular")};
            }

            TString Name_;
            int Offset_;
            int Length_;
            int Flags_;
        };

        TPortoClientPtr myPorto = new TMyPortoFailingClient();
        TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

        auto node = CreateNode(porto);
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL_C(ENodeStatus::SUCCESS, result.Success().Status, ToString(result.Success().Status));

        TString stdout = GetContainerStatus(Holder_->GetObjectStatus(WorkloadId_)).current().stdout_tail();
        UNIT_ASSERT_EQUAL_C(stdout, "[Porto client failed to get container stream]: 'InvalidData(6):::File is non-regular'", stdout);

        UNIT_ASSERT_EQUAL_C(ContainerName_, ((TMyPortoFailingClient*)myPorto.Get())->Name_, ((TMyPortoFailingClient*)myPorto.Get())->Name_);
        UNIT_ASSERT_EQUAL_C(Offset_, ((TMyPortoFailingClient*)myPorto.Get())->Offset_, ((TMyPortoFailingClient*)myPorto.Get())->Offset_);
        UNIT_ASSERT_EQUAL_C(Length_, ((TMyPortoFailingClient*)myPorto.Get())->Length_, ((TMyPortoFailingClient*)myPorto.Get())->Length_);
        UNIT_ASSERT_EQUAL_C(Flags_, ((TMyPortoFailingClient*)myPorto.Get())->Flags_, ((TMyPortoFailingClient*)myPorto.Get())->Flags_);
    }

private:
    virtual NStatusRepositoryTypes::TContainerDescription::EContainerType GetContainerType() const = 0;
    virtual API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const = 0;

    TFeedbackContainerStdoutNodePtr CreateNode(const TAsyncPortoClientPtr porto) {
        return new TFeedbackContainerStdoutNode(
            {1, "title"}
            , Holder_
            , porto
            , NStatusRepositoryTypes::TContainerDescription(
                WorkloadId_
                , NStatusRepositoryTypes::EObjectType::WORKLOAD
                , GetContainerType()
            )
            , ContainerName_
        );
    }

public:
    TWorkloadStatusRepositoryPtr Holder_;
    const TString WorkloadId_;
    const TString ContainerName_{"MyContainerName"};
    const int Offset_ = -1;
    const int Length_ = 1 << 12;
    const int Flags_ = 0;
};

class TFeedbackWorkloadContainerReadinessTestCase: public TFeedbackWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.readiness_status().container_status();
    }
};

class TFeedbackWorkloadContainerLivenessTestCase: public TFeedbackWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::LIVENESS;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.liveness_status().container_status();
    }
};

class TFeedbackWorkloadContainerStartTestCase: public TFeedbackWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::START;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.start();
    }
};

class TFeedbackWorkloadContainerStopTestCase: public TFeedbackWorkloadContainerBaseTestCase {
private:
    NStatusRepositoryTypes::TContainerDescription::EContainerType
    GetContainerType() const override {
        return NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP;
    }

    API::TContainerStatus GetContainerStatus(const API::TWorkloadStatus& status) const override {
        return status.stop_status().container_status();
    }
};

Y_UNIT_TEST_SUITE_F(FeedbackWorkloadContainerReadinessStdoutNodeSuite, TFeedbackWorkloadContainerReadinessTestCase) {

Y_UNIT_TEST(TestStdout) {
    TestStdout();
}

Y_UNIT_TEST(TestOnPortoError) {
    TestOnPortoError();
}

Y_UNIT_TEST(TestOnInvalidDataPortoError) {
    TestOnInvalidDataPortoError();
}

}

Y_UNIT_TEST_SUITE_F(FeedbackWorkloadContainerLivenessStdoutNodeSuite, TFeedbackWorkloadContainerLivenessTestCase) {

Y_UNIT_TEST(TestStdout) {
    TestStdout();
}

Y_UNIT_TEST(TestOnPortoError) {
    TestOnPortoError();
}

Y_UNIT_TEST(TestOnInvalidDataPortoError) {
    TestOnInvalidDataPortoError();
}

}

Y_UNIT_TEST_SUITE_F(FeedbackWorkloadContainerStartStdoutNodeSuite, TFeedbackWorkloadContainerStartTestCase) {

Y_UNIT_TEST(TestStdout) {
    TestStdout();
}

Y_UNIT_TEST(TestOnPortoError) {
    TestOnPortoError();
}

Y_UNIT_TEST(TestOnInvalidDataPortoError) {
    TestOnInvalidDataPortoError();
}

}

Y_UNIT_TEST_SUITE_F(FeedbackWorkloadContainerStopStdoutNodeSuite, TFeedbackWorkloadContainerStopTestCase) {

Y_UNIT_TEST(TestStdout) {
    TestStdout();
}

Y_UNIT_TEST(TestOnPortoError) {
    TestOnPortoError();
}

Y_UNIT_TEST(TestOnInvalidDataPortoError) {
    TestOnInvalidDataPortoError();
}

}

} // namespace NInfra::NPodAgent::NTestFeedbackContainerStdoutNode
