#include "feedback_object_check_failed_message_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/layer_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestFeedbackObjectCheckFailedMessageNode {

static TLogger logger({});

Y_UNIT_TEST_SUITE(FeedbackObjectCheckFailedMessageNodeSuite) {

TFeedbackObjectCheckFailedMessageNodePtr CreateNode(
    const TStatusRepositoryCommonPtr statusRepositoryCommon
    , const TString& objectIdOrHash
    , const TPortoContainerName& containerName
    , const TAsyncPortoClientPtr porto
    , const TString& ref
) {
    return new TFeedbackObjectCheckFailedMessageNode(
        {1, "title"}
        , statusRepositoryCommon
        , porto
        , objectIdOrHash
        , containerName
        , EPortoContainerProperty::Cwd
        , ref
    );
}

Y_UNIT_TEST(TestWhenWeHaveInvalidFailedMessage) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name = name;
            Property = property;
            return TString("222222");
        }

        TPortoContainerName Name = TPortoContainerName("");
        EPortoContainerProperty Property;
    };

    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    const TString id = "my_layer";
    const TString hash = "my_layer_dowonload_hash";
    TPortoContainerName containerName("container_name");
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, hash));
    TString ref = "11111";

    auto node = CreateNode(holder, hash, containerName, porto, ref);
    auto result = node->Tick(MockTickContext(logger));

    auto reason = "verify bad hash, expected '11111', got '222222'";
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    API::TLayerStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(reason, status.failed().message());
    UNIT_ASSERT_EQUAL(1, status.fail_counter());

    UNIT_ASSERT_EQUAL(((TMyPortoClient*)myPorto.Get())->Name, containerName);
    UNIT_ASSERT_EQUAL(((TMyPortoClient*)myPorto.Get())->Property, EPortoContainerProperty::Cwd);
}

Y_UNIT_TEST(TestWhenPortoReturnsError) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name = name;
            Property = property;
            return TPortoError{EPortoError::Unknown, TString("some error"), TString("")};
        }

        TPortoContainerName Name = TPortoContainerName("");
        EPortoContainerProperty Property;
    };

    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TString id = "my_layer";
    const TString hash = "my_layer_dowonload_hash";
    TPortoContainerName containerName("container_name");
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, hash));
    TString ref = "11111";

    auto node = CreateNode(holder, hash, containerName, porto, ref);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "some error");
    API::TLayerStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL("", status.failed().message());
    UNIT_ASSERT_EQUAL(0, status.fail_counter());
    UNIT_ASSERT_EQUAL(((TMyPortoFailingClient*)myPorto.Get())->Name, containerName);
    UNIT_ASSERT_EQUAL(((TMyPortoFailingClient*)myPorto.Get())->Property, EPortoContainerProperty::Cwd);
}

}

} // namespace NInfra::NPodAgent::NTestFeedbackObjectCheckFailedMessageNode
