#include "add_ip_address_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/ip_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NAddIpAddressNodeTest {

Y_UNIT_TEST_SUITE(AddIpAddressNodeSuite) {

static TLogger logger({});

TAddIpAddressNodePtr CreateNode(
    TIpClientPtr ipClient
    , const TString& device
    , const TIpDescription ipDescription
) {
    return new TAddIpAddressNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , new TAsyncIpClient(ipClient, new TFakeThreadPool())
        , device
        , ipDescription
    );
}

Y_UNIT_TEST(TestAddIpAddress) {
    struct TMyIpClient : public TMockIpClient {
        TExpected<void, TIpClientError> AddAddress(const TString& device, const TIpDescription& ip) override {
            ++Calls_;

            LastDevice_ = device;
            LastIp_ = ip;

            return TExpected<void, TIpClientError>::DefaultSuccess();
        }

        size_t Calls_ = 0;
        TString LastDevice_ = "";
        TIpDescription LastIp_ = TIpDescription("", 0);
    };

    TIpClientPtr ipClient = new TMyIpClient();

    const TString device = "my_device";
    const TIpDescription ipDescription = TIpDescription("ip", 128);

    auto node = CreateNode(ipClient, device, ipDescription);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    UNIT_ASSERT_EQUAL_C(1, ((TMyIpClient*)ipClient.Get())->Calls_, ((TMyIpClient*)ipClient.Get())->Calls_);
    UNIT_ASSERT_EQUAL_C(device, ((TMyIpClient*)ipClient.Get())->LastDevice_, ((TMyIpClient*)ipClient.Get())->LastDevice_);
    UNIT_ASSERT_EQUAL_C(ipDescription.Ip6, ((TMyIpClient*)ipClient.Get())->LastIp_.Ip6, ((TMyIpClient*)ipClient.Get())->LastIp_.Ip6);
    UNIT_ASSERT_EQUAL_C(ipDescription.Subnet, ((TMyIpClient*)ipClient.Get())->LastIp_.Subnet, ((TMyIpClient*)ipClient.Get())->LastIp_.Subnet);
}

Y_UNIT_TEST(TestAddIpAddressWithFailingClient) {
        struct TMyFailingIpClient : public TMockIpClient {
        TExpected<void, TIpClientError> AddAddress(const TString& device, const TIpDescription& ip) override {
            ++Calls_;

            LastDevice_ = device;
            LastIp_ = ip;

            return TIpClientError(EIpClientError::Unspecified, "AddIpAddress");
        }

        size_t Calls_ = 0;
        TString LastDevice_ = "";
        TIpDescription LastIp_ = TIpDescription("", 0);
    };

    TIpClientPtr ipClient = new TMyFailingIpClient();

    const TString device = "my_device";
    const TIpDescription ipDescription = TIpDescription("ip", 128);

    auto node = CreateNode(ipClient, device, ipDescription);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "AddIpAddress");

    UNIT_ASSERT_EQUAL_C(1, ((TMyFailingIpClient*)ipClient.Get())->Calls_, ((TMyFailingIpClient*)ipClient.Get())->Calls_);
    UNIT_ASSERT_EQUAL_C(device, ((TMyFailingIpClient*)ipClient.Get())->LastDevice_, ((TMyFailingIpClient*)ipClient.Get())->LastDevice_);
    UNIT_ASSERT_EQUAL_C(ipDescription.Ip6, ((TMyFailingIpClient*)ipClient.Get())->LastIp_.Ip6, ((TMyFailingIpClient*)ipClient.Get())->LastIp_.Ip6);
    UNIT_ASSERT_EQUAL_C(ipDescription.Subnet, ((TMyFailingIpClient*)ipClient.Get())->LastIp_.Subnet, ((TMyFailingIpClient*)ipClient.Get())->LastIp_.Subnet);
}

}

} // namespace NInfra::NPodAgent::NAddIpAddressNodeTest
