#include "capture_network_hook_status_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/common.h>

#include <infra/pod_agent/libs/common/common.h>

#include <infra/pod_agent/libs/system_logs_sender/system_logs_session_parameters.h>

namespace NInfra::NPodAgent {

ENodeType TCaptureNetworkHookStatusNode::GetType() const {
    return TCaptureNetworkHookStatusNode::NODE_TYPE;
}

TExpected<NThreading::TFuture<TExpected<void, TSystemLogsSenderError>>, TTickResult> TCaptureNetworkHookStatusNode::CaptureStatus(TTickContextPtr context) {
    API::THttpGetStatus::TAttemptFeedback httpFeedback;
    NLogEvent::THttpAttemptFeedback httpEvent;

    API::TTcpCheckStatus::TAttemptFeedback tcpFeedback;
    NLogEvent::TTcpAttemptFeedback tcpEvent;
    
    TString systemLogMessage;

    switch (HookBackend_) {
        case NStatusRepositoryTypes::EHookBackend::HTTP:
            httpFeedback = WorkloadStatusRepository_->CaptureHttpHookStatus(WorkloadId_, NetworkHookType_);
            httpEvent = CreateHttpLogEvent(httpFeedback);

            context.Get()->LogFrame.Get()->LogEvent(httpEvent);
            systemLogMessage = CreateHttpSystemLogMessage(httpEvent);
            break;
        case NStatusRepositoryTypes::EHookBackend::TCP:
            tcpFeedback = WorkloadStatusRepository_->CaptureTcpHookStatus(WorkloadId_, NetworkHookType_);
            tcpEvent = CreateTcpLogEvent(tcpFeedback);

            context.Get()->LogFrame.Get()->LogEvent(tcpEvent);
            systemLogMessage = CreateTcpSystemLogMessage(tcpEvent);
            break;
        case NStatusRepositoryTypes::EHookBackend::CONTAINER:
        case NStatusRepositoryTypes::EHookBackend::UNIX_SIGNAL:
        case NStatusRepositoryTypes::EHookBackend::NO_HOOK:
            ythrow yexception() << ToString(HookBackend_) << " shouldn't be provided";
    }

    return SystemLogsSender_->SendMessage(WorkloadId_, systemLogMessage);
}

NLogEvent::THttpAttemptFeedback TCaptureNetworkHookStatusNode::CreateHttpLogEvent(const API::THttpGetStatus::TAttemptFeedback& httpFeedback) {
    NLogEvent::THttpAttemptFeedback httpEvent;

    ui64 startTimeInMicroSeconds = TimestampToMicros(httpFeedback.start_time());
    ui64 deathTimeInMicroSeconds = TimestampToMicros(httpFeedback.death_time());

    httpEvent.set_hook_type(GetNetworkHookTypeFromEnum(NetworkHookType_));
    httpEvent.set_id(WorkloadId_);
    httpEvent.set_object_type(NLogEvent::EPodAgentObjectType::WORKLOAD);
    httpEvent.set_fail_reason(httpFeedback.fail_reason());
    httpEvent.set_inner_fail_reason(httpFeedback.inner_fail_reason());
    httpEvent.set_state((NLogEvent::THttpAttemptFeedback::EHttpGetState)httpFeedback.state());
    httpEvent.set_start_time(startTimeInMicroSeconds);
    httpEvent.set_death_time(deathTimeInMicroSeconds);
    
    return httpEvent;
}

TString TCaptureNetworkHookStatusNode::CreateHttpSystemLogMessage(const NLogEvent::THttpAttemptFeedback& httpEvent) {
    NLogEvent::THttpHookSystemLogsEvent httpHookSystemLogsEvent;

    httpHookSystemLogsEvent.SetloggerName(TSystemLogsSessionParameters::DEPLOY_LOGGER_NAME_VALUE);
    httpHookSystemLogsEvent.set_message(LogEventToMessage<NLogEvent::THttpAttemptFeedback>(httpEvent));
    httpHookSystemLogsEvent.set_workload(WorkloadId_);
    httpHookSystemLogsEvent.set_state(httpEvent.state());
    httpHookSystemLogsEvent.set_fail_reason(httpEvent.fail_reason());
    httpHookSystemLogsEvent.set_inner_fail_reason(httpEvent.inner_fail_reason());
    httpHookSystemLogsEvent.set_hook_type(httpEvent.hook_type());

    return LogEventToMessage<NLogEvent::THttpHookSystemLogsEvent>(httpHookSystemLogsEvent);
}


NLogEvent::TTcpAttemptFeedback TCaptureNetworkHookStatusNode::CreateTcpLogEvent(const API::TTcpCheckStatus::TAttemptFeedback& tcpFeedback) {
    NLogEvent::TTcpAttemptFeedback tcpEvent;

    ui64 startTimeInMicroSeconds = TimestampToMicros(tcpFeedback.start_time());
    ui64 deathTimeInMicroSeconds = TimestampToMicros(tcpFeedback.death_time());

    tcpEvent.set_hook_type(GetNetworkHookTypeFromEnum(NetworkHookType_));
    tcpEvent.set_id(WorkloadId_);
    tcpEvent.set_object_type(NLogEvent::EPodAgentObjectType::WORKLOAD);
    tcpEvent.set_fail_reason(tcpFeedback.fail_reason());
    tcpEvent.set_state((NLogEvent::TTcpAttemptFeedback::ETcpCheckState)tcpFeedback.state());
    tcpEvent.set_start_time(startTimeInMicroSeconds);
    tcpEvent.set_death_time(deathTimeInMicroSeconds);
    
    return tcpEvent;
}

TString TCaptureNetworkHookStatusNode::CreateTcpSystemLogMessage(const NLogEvent::TTcpAttemptFeedback& tcpEvent) {
    NLogEvent::TTcpHookSystemLogsEvent tcpHookSystemLogsEvent;

    tcpHookSystemLogsEvent.SetloggerName(TSystemLogsSessionParameters::DEPLOY_LOGGER_NAME_VALUE);
    tcpHookSystemLogsEvent.set_message(LogEventToMessage<NLogEvent::TTcpAttemptFeedback>(tcpEvent));
    tcpHookSystemLogsEvent.set_workload(WorkloadId_);
    tcpHookSystemLogsEvent.set_state(tcpEvent.state());
    tcpHookSystemLogsEvent.set_fail_reason(tcpEvent.fail_reason());
    tcpHookSystemLogsEvent.set_hook_type(tcpEvent.hook_type());

    return LogEventToMessage<NLogEvent::TTcpHookSystemLogsEvent>(tcpHookSystemLogsEvent);
}

TTickResult TCaptureNetworkHookStatusNode::ProcessResult(TTickContextPtr/* context*/, TExpected<void, TSystemLogsSenderError>& result) {
    if (!bool(result)) {
        return TNodeError({ToString(result.Error())});
    }   

    return TNodeSuccess(ENodeStatus::SUCCESS);
}

} // namespace NInfra::NPodAgent
