#include "capture_network_hook_status_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/test_functions.h>

#include <infra/pod_agent/libs/system_logs_sender/mock_system_logs_session.h>
#include <infra/pod_agent/libs/system_logs_sender/sidecars_system_logs_filter_impl.h>
#include <infra/pod_agent/libs/system_logs_sender/system_logs_sender_impl.h>
#include <infra/pod_agent/libs/system_logs_sender/system_logs_session_collection_impl.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestCaptureNetworkHookStatusNode {

Y_UNIT_TEST_SUITE(CaptureNetworkHookStatusNodeSuite) {

static TLogger logger({});

class TTestSystemLogsSession: public TMockSystemLogsSession
{
public:
    virtual ~TTestSystemLogsSession() = default;

    void SendMessage(const TString& message) override {
        SendMessageCalls_++;
        SentMessage_ = message;
    }

    TString SentMessage_ = "";
    int SendMessageCalls_ = 0;
};

Y_UNIT_TEST(TestCaptureHookStatus) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    TSystemLogsSenderPtr systemLogsSender = new TSystemLogsSender(new TSystemLogsSessionCollection, new TFakeThreadPool, new TSystemLogsStatistics, new TSidecarsSystemLogsFilterImpl());

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        TCaptureNetworkHookStatusNodePtr node = new TCaptureNetworkHookStatusNode(TBasicTreeNodeDescriptor{1, "title"}, holder, systemLogsSender, id, networkHookType, hookBackend);
        
        TSystemLogsSessionPtr systemLogsSession = new TTestSystemLogsSession;
        systemLogsSender->Add(id, systemLogsSession);

        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            holder->UpdateHttpHookState(id, API::EHttpGetState::EHttpGetState_RUNNING, networkHookType);
            holder->UpdateHttpHookFailReason(id, networkHookType, "fail_reason");
            holder->UpdateHttpHookInnerFailReason(id, networkHookType, "inner_fail_reason");
            UNIT_ASSERT(!GetHttpHookStatus(holder->GetObjectStatus(id), networkHookType).has_last());
        } else {
            holder->UpdateTcpCheckState(id, API::ETcpCheckState::ETcpCheckState_RUNNING, networkHookType);
            holder->UpdateTcpCheckFailReason(id, networkHookType, "fail_reason");
            UNIT_ASSERT(!GetTcpCheckStatus(holder->GetObjectStatus(id), networkHookType).has_last());
        }

        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(TNodeSuccess(ENodeStatus::SUCCESS), result.Success());
        UNIT_ASSERT_EQUAL(((TTestSystemLogsSession*)systemLogsSession.Get())->SendMessageCalls_, 1);

        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            auto status = GetHttpHookStatus(holder->GetObjectStatus(id), networkHookType).last();
            UNIT_ASSERT_EQUAL(API::EHttpGetState::EHttpGetState_RUNNING, status.state());
            UNIT_ASSERT_EQUAL("fail_reason", status.fail_reason());
            UNIT_ASSERT_EQUAL("inner_fail_reason", status.inner_fail_reason());

            auto logEvent = node->CreateHttpLogEvent(status);
            auto systemLogsMessage = node->CreateHttpSystemLogMessage(logEvent);       
            UNIT_ASSERT_EQUAL(((TTestSystemLogsSession*)systemLogsSession.Get())->SentMessage_, systemLogsMessage);

            status = GetHttpHookStatus(holder->GetObjectStatus(id), networkHookType).current();
            UNIT_ASSERT_EQUAL(API::EHttpGetState::EHttpGetState_UNKNOWN, status.state());
            UNIT_ASSERT_EQUAL("", status.fail_reason());
            UNIT_ASSERT_EQUAL("", status.inner_fail_reason());
        } else {
            auto status = GetTcpCheckStatus(holder->GetObjectStatus(id), networkHookType).last();
            UNIT_ASSERT_EQUAL(API::ETcpCheckState::ETcpCheckState_RUNNING, status.state());
            UNIT_ASSERT_EQUAL("fail_reason", status.fail_reason());

            auto logEvent = node->CreateTcpLogEvent(status);
            auto systemLogsMessage = node->CreateTcpSystemLogMessage(logEvent);       
            UNIT_ASSERT_EQUAL(((TTestSystemLogsSession*)systemLogsSession.Get())->SentMessage_, systemLogsMessage);

            status = GetTcpCheckStatus(holder->GetObjectStatus(id), networkHookType).current();
            UNIT_ASSERT_EQUAL(API::ETcpCheckState::ETcpCheckState_UNKNOWN, status.state());
            UNIT_ASSERT_EQUAL("", status.fail_reason());
        }

        node = new TCaptureNetworkHookStatusNode(TBasicTreeNodeDescriptor{1, "title"}, holder, systemLogsSender, id, networkHookType, hookBackend);
        systemLogsSender->RemoveAll();
        result = node->Tick(MockTickContext(logger));
        UNIT_ASSERT(!(bool)result);
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestWithUndefinedStatusRepositoryHttp) {
    TSystemLogsSenderPtr systemLogsSender = new TSystemLogsSender(new TSystemLogsSessionCollection, new TFakeThreadPool, new TSystemLogsStatistics, new TSidecarsSystemLogsFilterImpl());

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TCaptureNetworkHookStatusNode(
                TBasicTreeNodeDescriptor{1, "title"}
                , nullptr
                , systemLogsSender
                , "my_workload"
                , networkHookType
                , hookBackend
            )
            , yexception
            , "WorkloadStatusRepository not defined"
        );
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestWithUndefinedSystemLogsSenderHttp) {
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TCaptureNetworkHookStatusNode(
                TBasicTreeNodeDescriptor{1, "title"}
                , statusRepository
                , nullptr
                , "my_workload"
                , networkHookType
                , hookBackend
            )
            , yexception
            , "system logs sender not defined"
        );
    };

    TestNetworkHooksAllTypes(test);
}

}

} // namespace NInfra::NPodAgent::NTestCaptureNetworkHookStatusNode
