#include "check_http_hook_retries_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestCheckHttpHookRetriesNode  {

static TLogger logger({});

TCheckHttpHookRetriesNodePtr GetCheckHttpHookRetriesNode(
    TWorkloadStatusRepositoryPtr statusRepository
    , const TString& workloadId
    , ui32 maxTries
    , NStatusRepositoryTypes::ENetworkHookType networkHookType
) {
    if (statusRepository) {
        statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    }

    return new TCheckHttpHookRetriesNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , statusRepository
        , workloadId
        , networkHookType
        , maxTries
    );
}

Y_UNIT_TEST_SUITE(CheckHttpHookRetriesNode) {

Y_UNIT_TEST(TestSuccess) {
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    const TString workloadId = "my_workload";
    auto node = GetCheckHttpHookRetriesNode(statusRepository, workloadId, 2, NStatusRepositoryTypes::ENetworkHookType::DESTROY);
    statusRepository->IncrementHttpRequestsSuccessCounter(workloadId, NStatusRepositoryTypes::ENetworkHookType::DESTROY);
    auto res = node->Tick(MockTickContext(logger));

    UNIT_ASSERT((bool)res);
    UNIT_ASSERT_EQUAL(res.Success().Status, ENodeStatus::SUCCESS);
}

Y_UNIT_TEST(TestFailure) {
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    const TString workloadId = "my_workload";
    auto node = GetCheckHttpHookRetriesNode(statusRepository, workloadId, 4, NStatusRepositoryTypes::ENetworkHookType::DESTROY);
    statusRepository->IncrementHttpRequestsSuccessCounter(workloadId, NStatusRepositoryTypes::ENetworkHookType::DESTROY);
    statusRepository->IncrementHttpRequestsErrorCounter(workloadId, NStatusRepositoryTypes::ENetworkHookType::DESTROY);
    statusRepository->IncrementHttpRequestsTimeoutCounter(workloadId, NStatusRepositoryTypes::ENetworkHookType::DESTROY);
    statusRepository->IncrementHttpRequestsWrongAnswerCounter(workloadId, NStatusRepositoryTypes::ENetworkHookType::DESTROY);
    auto res = node->Tick(MockTickContext(logger));

    UNIT_ASSERT((bool)res);
    UNIT_ASSERT_EQUAL(res.Success().Status, ENodeStatus::FAILURE);
    UNIT_ASSERT_STRING_CONTAINS(TStringBuilder() << "Workload '" << workloadId << "' exceeded destroy tries limit: 4", res.Success().Message);
}

Y_UNIT_TEST(TestWorkloadStatusRepositoryNotDefined) {
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        GetCheckHttpHookRetriesNode(
            nullptr
            , "my_workload"
            , 1
            , NStatusRepositoryTypes::ENetworkHookType::DESTROY
        )
        , yexception
        , "WorkloadStatusRepository not defined"
    );
}

Y_UNIT_TEST(TestStatusMaxTriesIsZero) {
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    auto node = GetCheckHttpHookRetriesNode(statusRepository, "my_workload", 0, NStatusRepositoryTypes::ENetworkHookType::DESTROY);
    auto res = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_STRING_CONTAINS(res.Error().Message, "MaxTries is zero");
}

}

} // namespace NInfra::NPodAgent::NTestCheckHttpHookRetriesNode
