#include "feedback_network_hook_state_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/test_functions.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestFeedbackNetworkHookStateNode {

Y_UNIT_TEST_SUITE(FeedbackNetworkHookStateNodeSuite) {

static TLogger logger({});

Y_UNIT_TEST(TestUpdateState) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        TFeedbackNetworkHookStateNode::TNetworkHookState state;
        if (hookBackend == NStatusRepositoryTypes::EHookBackend::TCP) {
            state = API::ETcpCheckState_WAITING_RESTART;
        } else {
            state = API::EHttpGetState_WAITING_RESTART;
        }

        TFeedbackNetworkHookStateNodePtr node = new TFeedbackNetworkHookStateNode(
            TBasicTreeNodeDescriptor{1, "title"}
            , holder
            , id
            , networkHookType
            , hookBackend
            , state
        );

        auto result = node->Tick(MockTickContext(logger));
        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(TNodeSuccess(ENodeStatus::SUCCESS), result.Success());

        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            auto status = GetHttpHookStatus(holder->GetObjectStatus(id), networkHookType);
            UNIT_ASSERT_EQUAL(std::get<API::EHttpGetState>(state), status.current().state());
        } else {
            auto status = GetTcpCheckStatus(holder->GetObjectStatus(id), networkHookType);
            UNIT_ASSERT_EQUAL(std::get<API::ETcpCheckState>(state), status.current().state());
        }
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestWithUndefinedWorkloadStatusRepository) {
    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        TFeedbackNetworkHookStateNode::TNetworkHookState state;
        if (hookBackend == NStatusRepositoryTypes::EHookBackend::TCP) {
            state = API::ETcpCheckState_WAITING_RESTART;
        } else {
            state = API::EHttpGetState_WAITING_RESTART;
        }

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TFeedbackNetworkHookStateNode(
                TBasicTreeNodeDescriptor{1, "title"}
                , nullptr
                , "my_workload"
                , networkHookType
                , hookBackend
                , state
            )
            , yexception
            , "WorkloadStatusRepository not defined"
        );
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestWithUnexpectedStateType) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        TFeedbackNetworkHookStateNode(
            TBasicTreeNodeDescriptor{1, "title"}
            , holder
            , "my_workload"
            , NStatusRepositoryTypes::ENetworkHookType::READINESS
            , NStatusRepositoryTypes::EHookBackend::HTTP
            , API::ETcpCheckState_WAITING_RESTART
        )
        , yexception
        , "expected EHttpGetState"
    );
}

}

} // namespace NInfra::NPodAgent::NTestFeedbackNetworkHookStateNode
