#include "network_check_and_add_request_node.h"

namespace NInfra::NPodAgent {

ENodeType TNetworkCheckAndAddRequestNode::GetType() const {
    return TNetworkCheckAndAddRequestNode::NODE_TYPE;
}

TExpected<void, TNetworkClientError> TNetworkCheckAndAddRequestNode::NetworkClientCall(TTickContextPtr /*context*/) {
    switch (HookBackend_) {
        case NStatusRepositoryTypes::EHookBackend::HTTP:
            return NetworkClient_->CheckAndAddHttpRequest(
                GetRequestKey()
                , RequestHash_
                , WorkloadId_
                , "localhost"
                , Port_
                , Path_
                , Timeout_
            );
            break;
        case NStatusRepositoryTypes::EHookBackend::TCP:
            return NetworkClient_->CheckAndAddTcpRequest(
                GetRequestKey()
                , RequestHash_
                , WorkloadId_
                , Port_
                , Timeout_
            );
            break;
        case NStatusRepositoryTypes::EHookBackend::CONTAINER:
        case NStatusRepositoryTypes::EHookBackend::UNIX_SIGNAL:
        case NStatusRepositoryTypes::EHookBackend::NO_HOOK:
            ythrow yexception() << ToString(HookBackend_) << " shouldn't be provided";
    }
}

TTickResult TNetworkCheckAndAddRequestNode::ProcessNetworkClientResult(TTickContextPtr /*context*/, TExpected<void, TNetworkClientError>& result) {
    if (!result && result.Error().Errno == ENetworkClientError::RequestAlreadyExist) {
        return TNodeSuccess(ENodeStatus::SUCCESS, "Request already exist");
    }

    switch (HookBackend_) {
        case NStatusRepositoryTypes::EHookBackend::HTTP:
            WorkloadStatusRepository_->UpdateHttpHookStartTime(WorkloadId_, TInstant::Now(), NetworkHookType_);
            if (result) {
                WorkloadStatusRepository_->UpdateHttpHookState(WorkloadId_, API::EHttpGetState::EHttpGetState_RUNNING, NetworkHookType_);
                return TNodeSuccess(ENodeStatus::SUCCESS);
            } else {
                WorkloadStatusRepository_->UpdateHttpHookState(WorkloadId_, API::EHttpGetState::EHttpGetState_FAILURE, NetworkHookType_);
                WorkloadStatusRepository_->UpdateHttpHookInnerFailReason(WorkloadId_, NetworkHookType_, MakeFullErrorMessage(result.Error().Message));
                WorkloadStatusRepository_->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(WorkloadId_, NetworkHookType_, false);

                WorkloadStatusRepository_->IncrementHttpRequestsErrorCounter(WorkloadId_, NetworkHookType_);

                return TNodeSuccess{
                    ENodeStatus::FAILURE
                    , ToString(result.Error())
                };
            }
            break;
        case NStatusRepositoryTypes::EHookBackend::TCP:
            WorkloadStatusRepository_->UpdateTcpCheckStartTime(WorkloadId_, TInstant::Now(), NetworkHookType_);
            if (result) {
                WorkloadStatusRepository_->UpdateTcpCheckState(WorkloadId_, API::ETcpCheckState::ETcpCheckState_RUNNING, NetworkHookType_);
                return TNodeSuccess(ENodeStatus::SUCCESS);
            } else {
                WorkloadStatusRepository_->UpdateTcpCheckState(WorkloadId_, API::ETcpCheckState::ETcpCheckState_FAILURE, NetworkHookType_);
                WorkloadStatusRepository_->UpdateTcpCheckFailReason(WorkloadId_, NetworkHookType_, MakeFullErrorMessage(result.Error().Message));
                WorkloadStatusRepository_->UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(WorkloadId_, NetworkHookType_, false);

                WorkloadStatusRepository_->IncrementTcpCheckErrorCounter(WorkloadId_, NetworkHookType_);

                return TNodeSuccess{
                    ENodeStatus::FAILURE
                    , ToString(result.Error())
                };
            }
            break;
        case NStatusRepositoryTypes::EHookBackend::CONTAINER:
        case NStatusRepositoryTypes::EHookBackend::UNIX_SIGNAL:
        case NStatusRepositoryTypes::EHookBackend::NO_HOOK:
            ythrow yexception() << ToString(HookBackend_) << " shouldn't be provided";
    }
}

} // namespace NInfra::NPodAgent
