#include "network_check_and_add_request_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/network_client/mock_client.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/test_functions.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestNetworkCheckAndAddRequestNode {

static TLogger logger({});

TNetworkCheckAndAddRequestNodePtr CreateNode(
    TWorkloadStatusRepositoryPtr workloadStatusRepository
    , const TString& workloadId
    , NStatusRepositoryTypes::ENetworkHookType networkHookType
    , NStatusRepositoryTypes::EHookBackend hookBackend
    , const TString& requestHash
    , TNetworkClientPtr networkClient
    , const ui32 port
    , const TString& path
    , const TDuration& timeout
) {
    return new TNetworkCheckAndAddRequestNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , workloadStatusRepository
        , workloadId
        , networkHookType
        , hookBackend
        , requestHash
        , networkClient
        , port
        , path
        , timeout
    );
}

Y_UNIT_TEST_SUITE(NetworkCheckAndAddRequestNodeSuite) {

Y_UNIT_TEST(TestCheckAndAddSuccess) {
    struct TMyNetworkClient : public TMockNetworkClient {
        TExpected<void, TNetworkClientError> CheckAndAddHttpRequest(
            const TString& requestKey
            , const TString& requestHash
            , const TString& additionalInfo
            , const TString& host
            , ui32 port
            , const TString& path
            , TDuration timeout
        ) override {
            RequestKey_ = requestKey;
            RequestHash_ = requestHash;
            AdditionalInfo_ = additionalInfo;
            Host_ = host;
            Port_ = port;
            Path_ = path;
            Timeout_ = timeout;
            return TExpected<void, TNetworkClientError>::DefaultSuccess();
        }

        TExpected<void, TNetworkClientError> CheckAndAddTcpRequest(
            const TString& requestKey
            , const TString& requestHash
            , const TString& additionalInfo
            , ui32 port
            , TDuration timeout
        ) override {
            RequestKey_ = requestKey;
            RequestHash_ = requestHash;
            AdditionalInfo_ = additionalInfo;
            Port_ = port;
            Path_ = "";
            Timeout_ = timeout;
            return TExpected<void, TNetworkClientError>::DefaultSuccess();
        }

        TString RequestKey_;
        TString RequestHash_;
        TString AdditionalInfo_;
        TString Host_;
        ui32 Port_;
        TString Path_;
        TDuration Timeout_;
    };

    const TString workloadId = "my_workload";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMyNetworkClient();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        const TString requestHash = "hash";
        const TString host = "localhost";
        const ui32 port = 1234;
        const TString path = hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP ? "/path" : "";
        const TDuration timeout = TDuration::Seconds(1234);
        const TString requestKey = workloadId + "_" + ToString(networkHookType);
        auto node = CreateNode(
            workloadStatusRepository
            , workloadId
            , networkHookType
            , hookBackend
            , requestHash
            , networkClient
            , port
            , path
            , timeout
        );
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpRequestsSuccessCounter(workloadId, networkHookType), 0);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpRequestsErrorCounter(workloadId, networkHookType), 0);
            UNIT_ASSERT_EQUAL(GetHttpHookStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::EHttpGetState::EHttpGetState_RUNNING);
        } else {
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetTcpCheckSuccessCounter(workloadId, networkHookType), 0);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetTcpCheckErrorCounter(workloadId, networkHookType), 0);
            UNIT_ASSERT_EQUAL(GetTcpCheckStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::ETcpCheckState::ETcpCheckState_RUNNING);
        }

        UNIT_ASSERT_STRINGS_EQUAL(requestKey, ((TMyNetworkClient*)networkClient.Get())->RequestKey_);
        UNIT_ASSERT_STRINGS_EQUAL(requestHash, ((TMyNetworkClient*)networkClient.Get())->RequestHash_);
        UNIT_ASSERT_STRINGS_EQUAL(workloadId, ((TMyNetworkClient*)networkClient.Get())->AdditionalInfo_);
        UNIT_ASSERT_STRINGS_EQUAL(host, ((TMyNetworkClient*)networkClient.Get())->Host_);
        UNIT_ASSERT_EQUAL(port, ((TMyNetworkClient*)networkClient.Get())->Port_);
        UNIT_ASSERT_STRINGS_EQUAL(path, ((TMyNetworkClient*)networkClient.Get())->Path_);
        UNIT_ASSERT_EQUAL(timeout, ((TMyNetworkClient*)networkClient.Get())->Timeout_);
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestCheckAndAddSuccessAlreadyExist) {
    struct TMyNetworkClient : public TMockNetworkClient {
        TExpected<void, TNetworkClientError> CheckAndAddHttpRequest(
            const TString& requestKey
            , const TString& requestHash
            , const TString& additionalInfo
            , const TString& host
            , ui32 port
            , const TString& path
            , TDuration timeout
        ) override {
            RequestKey_ = requestKey;
            RequestHash_ = requestHash;
            AdditionalInfo_ = additionalInfo;
            Host_ = host;
            Port_ = port;
            Path_ = path;
            Timeout_ = timeout;
            return TNetworkClientError(
                ENetworkClientError::RequestAlreadyExist
                , "already exist"
            );
        }

        TExpected<void, TNetworkClientError> CheckAndAddTcpRequest(
            const TString& requestKey
            , const TString& requestHash
            , const TString& additionalInfo
            , ui32 port
            , TDuration timeout
        ) override {
            RequestKey_ = requestKey;
            RequestHash_ = requestHash;
            AdditionalInfo_ = additionalInfo;
            Port_ = port;
            Path_ = "";
            Timeout_ = timeout;
            return TNetworkClientError(
                ENetworkClientError::RequestAlreadyExist
                , "already exist"
            );
        }

        TString RequestKey_;
        TString RequestHash_;
        TString AdditionalInfo_;
        TString Host_;
        ui32 Port_;
        TString Path_;
        TDuration Timeout_;
    };

    const TString workloadId = "my_workload";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMyNetworkClient();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        const TString requestHash = "hash";
        const TString host = "localhost";
        const ui32 port = 1234;
        const TString path = hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP ? "/path" : "";
        const TDuration timeout = TDuration::Seconds(1234);
        const TString requestKey = workloadId + "_" + ToString(networkHookType);
        auto node = CreateNode(
            workloadStatusRepository
            , workloadId
            , networkHookType
            , hookBackend
            , requestHash
            , networkClient
            , port
            , path
            , timeout
        );
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
        UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "Request already exist");

        // Nothing changes
        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpRequestsSuccessCounter(workloadId, networkHookType), 0);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpRequestsErrorCounter(workloadId, networkHookType), 0);
            UNIT_ASSERT_EQUAL(GetHttpHookStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::EHttpGetState::EHttpGetState_UNKNOWN);
        } else {
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetTcpCheckSuccessCounter(workloadId, networkHookType), 0);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetTcpCheckErrorCounter(workloadId, networkHookType), 0);
            UNIT_ASSERT_EQUAL(GetTcpCheckStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::ETcpCheckState::ETcpCheckState_UNKNOWN);
        }

        UNIT_ASSERT_STRINGS_EQUAL(requestKey, ((TMyNetworkClient*)networkClient.Get())->RequestKey_);
        UNIT_ASSERT_STRINGS_EQUAL(requestHash, ((TMyNetworkClient*)networkClient.Get())->RequestHash_);
        UNIT_ASSERT_STRINGS_EQUAL(workloadId, ((TMyNetworkClient*)networkClient.Get())->AdditionalInfo_);
        UNIT_ASSERT_STRINGS_EQUAL(host, ((TMyNetworkClient*)networkClient.Get())->Host_);
        UNIT_ASSERT_EQUAL(port, ((TMyNetworkClient*)networkClient.Get())->Port_);
        UNIT_ASSERT_STRINGS_EQUAL(path, ((TMyNetworkClient*)networkClient.Get())->Path_);
        UNIT_ASSERT_EQUAL(timeout, ((TMyNetworkClient*)networkClient.Get())->Timeout_);
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestCheckAndAddError) {
    struct TMyNetworkClient : public TMockNetworkClient {
        TExpected<void, TNetworkClientError> CheckAndAddHttpRequest(
            const TString& requestKey
            , const TString& requestHash
            , const TString& additionalInfo
            , const TString& host
            , ui32 port
            , const TString& path
            , TDuration timeout
        ) override {
            RequestKey_ = requestKey;
            RequestHash_ = requestHash;
            AdditionalInfo_ = additionalInfo;
            Host_ = host;
            Port_ = port;
            Path_ = path;
            Timeout_ = timeout;
            return TNetworkClientError(
                ENetworkClientError::Unspecified
                , "internal error"
            );
        }

        TExpected<void, TNetworkClientError> CheckAndAddTcpRequest(
            const TString& requestKey
            , const TString& requestHash
            , const TString& additionalInfo
            , ui32 port
            , TDuration timeout
        ) override {
            RequestKey_ = requestKey;
            RequestHash_ = requestHash;
            AdditionalInfo_ = additionalInfo;
            Port_ = port;
            Path_ = "";
            Timeout_ = timeout;
            return TNetworkClientError(
                ENetworkClientError::Unspecified
                , "internal error"
            );
        }

        TString RequestKey_;
        TString RequestHash_;
        TString AdditionalInfo_;
        TString Host_;
        ui32 Port_;
        TString Path_;
        TDuration Timeout_;
    };

    const TString workloadId = "my_workload";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMyNetworkClient();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        const TString requestHash = "hash";
        const TString host = "localhost";
        const ui32 port = 1234;
        const TString path = hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP ? "/path" : "";
        const TDuration timeout = TDuration::Seconds(1234);
        const TString requestKey = workloadId + "_" + ToString(networkHookType);
        auto node = CreateNode(
            workloadStatusRepository
            , workloadId
            , networkHookType
            , hookBackend
            , requestHash
            , networkClient
            , port
            , path
            , timeout
        );
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
        UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "internal error");

        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpHookConsecutiveFailuresCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpRequestsErrorCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpRequestsSuccessCounter(workloadId, networkHookType), 0);
            UNIT_ASSERT_EQUAL(GetHttpHookStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::EHttpGetState::EHttpGetState_FAILURE);
        } else {
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetTcpCheckConsecutiveFailuresCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetTcpCheckErrorCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetTcpCheckSuccessCounter(workloadId, networkHookType), 0);
            UNIT_ASSERT_EQUAL(GetTcpCheckStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::ETcpCheckState::ETcpCheckState_FAILURE);
        }

        UNIT_ASSERT_STRINGS_EQUAL(requestKey, ((TMyNetworkClient*)networkClient.Get())->RequestKey_);
        UNIT_ASSERT_STRINGS_EQUAL(requestHash, ((TMyNetworkClient*)networkClient.Get())->RequestHash_);
        UNIT_ASSERT_STRINGS_EQUAL(workloadId, ((TMyNetworkClient*)networkClient.Get())->AdditionalInfo_);
        UNIT_ASSERT_STRINGS_EQUAL(host, ((TMyNetworkClient*)networkClient.Get())->Host_);
        UNIT_ASSERT_EQUAL(port, ((TMyNetworkClient*)networkClient.Get())->Port_);
        UNIT_ASSERT_STRINGS_EQUAL(path, ((TMyNetworkClient*)networkClient.Get())->Path_);
        UNIT_ASSERT_EQUAL(timeout, ((TMyNetworkClient*)networkClient.Get())->Timeout_);
    };

    TestNetworkHooksAllTypes(test);
}

}

} // namespace NInfra::NPodAgent::NTestNetworkCheckAndAddRequestNode
