#include "network_get_and_remove_request_response_node.h"

namespace NInfra::NPodAgent {

ENodeType TNetworkGetAndRemoveRequestResponseNode::GetType() const {
    return TNetworkGetAndRemoveRequestResponseNode::NODE_TYPE;
}

TExpected<TString, TNetworkClientError> TNetworkGetAndRemoveRequestResponseNode::NetworkClientCall(TTickContextPtr /*context*/) {
    if (RequestDescription_.empty()) {
        RequestDescription_ = OUTCOME_TRYX(NetworkClient_->GetRequestDescription(GetRequestKey(), RequestHash_));
    }
    return NetworkClient_->GetAndRemoveRequestResponse(GetRequestKey(), RequestHash_);
}

TTickResult TNetworkGetAndRemoveRequestResponseNode::ProcessNetworkClientResult(TTickContextPtr /*context*/, TExpected<TString, TNetworkClientError>& result) {
    switch (HookBackend_) {
        case NStatusRepositoryTypes::EHookBackend::HTTP:
            // Increment request count only after request end
            WorkloadStatusRepository_->UpdateHttpHookDeathTime(WorkloadId_, TInstant::Now(), NetworkHookType_);

            if (!result) {
                WorkloadStatusRepository_->UpdateHttpHookState(WorkloadId_, API::EHttpGetState::EHttpGetState_FAILURE, NetworkHookType_);
                WorkloadStatusRepository_->UpdateHttpHookInnerFailReason(WorkloadId_, NetworkHookType_, MakeFullErrorMessage(result.Error().Message));
                WorkloadStatusRepository_->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(WorkloadId_, NetworkHookType_, false);
                if (result.Error().Errno == ENetworkClientError::RequestTimeout) {
                    WorkloadStatusRepository_->IncrementHttpRequestsTimeoutCounter(WorkloadId_, NetworkHookType_);
                } else {
                    WorkloadStatusRepository_->IncrementHttpRequestsErrorCounter(WorkloadId_, NetworkHookType_);
                }

                return TNodeSuccess({ENodeStatus::FAILURE, ToString(result.Error())});
            }

            if (!AnyAnswer_) {
                const TString& answer = result.Success();
                if (answer != ExpectedAnswer_) {
                    WorkloadStatusRepository_->UpdateHttpHookState(WorkloadId_, API::EHttpGetState::EHttpGetState_WRONG_ANSWER, NetworkHookType_);
                    WorkloadStatusRepository_->IncrementHttpRequestsWrongAnswerCounter(WorkloadId_, NetworkHookType_);
                    TStringBuilder failReason;
                    failReason << RequestDescription_ << ": wrong answer: expected '" << ExpectedAnswer_ << "' got '" << answer << "'";
                    WorkloadStatusRepository_->UpdateHttpHookFailReason(WorkloadId_, NetworkHookType_, MakeFullErrorMessage(failReason));
                    WorkloadStatusRepository_->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(WorkloadId_, NetworkHookType_, false);

                    return TNodeSuccess({ENodeStatus::FAILURE, failReason});
                }
            }

            WorkloadStatusRepository_->UpdateHttpHookState(WorkloadId_, API::EHttpGetState::EHttpGetState_SUCCESS, NetworkHookType_);
            WorkloadStatusRepository_->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(WorkloadId_, NetworkHookType_, true);
            WorkloadStatusRepository_->IncrementHttpRequestsSuccessCounter(WorkloadId_, NetworkHookType_);
            break;
        case NStatusRepositoryTypes::EHookBackend::TCP:
            WorkloadStatusRepository_->UpdateTcpCheckDeathTime(WorkloadId_, TInstant::Now(), NetworkHookType_);

            if (!result) {
                WorkloadStatusRepository_->UpdateTcpCheckState(WorkloadId_, API::ETcpCheckState::ETcpCheckState_FAILURE, NetworkHookType_);
                WorkloadStatusRepository_->UpdateTcpCheckFailReason(WorkloadId_, NetworkHookType_, MakeFullErrorMessage(result.Error().Message));
                WorkloadStatusRepository_->UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(WorkloadId_, NetworkHookType_, false);
                if (result.Error().Errno == ENetworkClientError::RequestTimeout) {
                    WorkloadStatusRepository_->IncrementTcpCheckTimeoutCounter(WorkloadId_, NetworkHookType_);
                } else {
                    WorkloadStatusRepository_->IncrementTcpCheckErrorCounter(WorkloadId_, NetworkHookType_);
                }

                return TNodeSuccess({ENodeStatus::FAILURE, ToString(result.Error())});
            }

            WorkloadStatusRepository_->UpdateTcpCheckState(WorkloadId_, API::ETcpCheckState::ETcpCheckState_SUCCESS, NetworkHookType_);
            WorkloadStatusRepository_->UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(WorkloadId_, NetworkHookType_, true);
            WorkloadStatusRepository_->IncrementTcpCheckSuccessCounter(WorkloadId_, NetworkHookType_);
            break;
        case NStatusRepositoryTypes::EHookBackend::CONTAINER:
        case NStatusRepositoryTypes::EHookBackend::UNIX_SIGNAL:
        case NStatusRepositoryTypes::EHookBackend::NO_HOOK:
            ythrow yexception() << ToString(HookBackend_) << " shouldn't be provided";
    }

    return TNodeSuccess(ENodeStatus::SUCCESS, ToString(result.Success()));
}

} // namespace NInfra::NPodAgent
