#include "network_get_and_remove_request_response_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/network_client/mock_client.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/test_functions.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestNetworkGetAndRemoveRequestResponseNode {

static TLogger logger({});

TNetworkGetAndRemoveRequestResponseNodePtr CreateNode(
    TWorkloadStatusRepositoryPtr workloadStatusRepository
    , const TString& workloadId
    , NStatusRepositoryTypes::ENetworkHookType networkHookType
    , NStatusRepositoryTypes::EHookBackend hookBackend
    , const TString& requestHash
    , TNetworkClientPtr networkClient
    , const TString& expectedAnswer
    , bool anyAnswer
) {
    return new TNetworkGetAndRemoveRequestResponseNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , workloadStatusRepository
        , workloadId
        , networkHookType
        , hookBackend
        , requestHash
        , networkClient
        , expectedAnswer
        , anyAnswer
    );
}

Y_UNIT_TEST_SUITE(NetworkGetAndRemoveRequestResponseNodeSuite) {

Y_UNIT_TEST(TestGetAndRemoveWithExpectedAnswer) {
    struct TMyNetworkClient : public TMockNetworkClient {
        TExpected<TString, TNetworkClientError> GetAndRemoveRequestResponse(const TString& requestKey, const TString& requestHash) override {
            RequestKey_ = requestKey;
            RequestHash_ = requestHash;
            return TString("expected_answer");
        }

        TString RequestKey_;
        TString RequestHash_;
    };

    const TString workloadId = "my_workload";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMyNetworkClient();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        const TString requestHash = "hash";
        const TString expectedAnswer = hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP ? "expected_answer": "";
        const TString requestKey = workloadId + "_" + ToString(networkHookType);
        auto node = CreateNode(
            workloadStatusRepository
            , workloadId
            , networkHookType
            , hookBackend
            , requestHash
            , networkClient
            , expectedAnswer
            , false
        );
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpRequestsSuccessCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpHookConsecutiveSuccessesCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(GetHttpHookStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::EHttpGetState::EHttpGetState_SUCCESS);
        } else {
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetTcpCheckSuccessCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetTcpCheckConsecutiveSuccessesCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(GetTcpCheckStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::ETcpCheckState::ETcpCheckState_SUCCESS);
        }

        UNIT_ASSERT_EQUAL(requestKey, ((TMyNetworkClient*)networkClient.Get())->RequestKey_);
        UNIT_ASSERT_EQUAL(requestHash, ((TMyNetworkClient*)networkClient.Get())->RequestHash_);
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestGetAndRemoveHttpWithAnyAnswer) {
    struct TMyNetworkClient : public TMockNetworkClient {
        TExpected<TString, TNetworkClientError> GetAndRemoveRequestResponse(const TString& requestKey, const TString& requestHash) override {
            RequestKey_ = requestKey;
            RequestHash_ = requestHash;
            return TString("some_answer");
        }

        TString RequestKey_;
        TString RequestHash_;
    };

    const TString workloadId = "my_workload";
    const TString requestHash = "hash";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMyNetworkClient();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType) {
        const TString requestKey = workloadId + "_" + ToString(networkHookType);
        auto node = CreateNode(
            workloadStatusRepository
            , workloadId
            , networkHookType
            , NStatusRepositoryTypes::EHookBackend::HTTP
            , requestHash
            , networkClient
            , ""
            , true
        );
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

        UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpRequestsSuccessCounter(workloadId, networkHookType), 1);
        UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpHookConsecutiveSuccessesCounter(workloadId, networkHookType), 1);
        UNIT_ASSERT_EQUAL(GetHttpHookStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::EHttpGetState::EHttpGetState_SUCCESS);

        UNIT_ASSERT_EQUAL(requestKey, ((TMyNetworkClient*)networkClient.Get())->RequestKey_);
        UNIT_ASSERT_EQUAL(requestHash, ((TMyNetworkClient*)networkClient.Get())->RequestHash_);
    };

    TestHttpHookAllTypes(test);
}

Y_UNIT_TEST(TestGetAndRemoveHttpWithWrongAnswer) {
    struct TMyNetworkClient : public TMockNetworkClient {
        TExpected<TString, TNetworkClientError> GetAndRemoveRequestResponse(const TString& requestKey, const TString& requestHash) override {
            RequestKey_ = requestKey;
            RequestHash_ = requestHash;
            return TString("wrong_answer");
        }

        TString RequestKey_;
        TString RequestHash_;
    };

    const TString workloadId = "my_workload";
    const TString requestHash = "hash";
    const TString expectedAnswer = "expected_answer";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMyNetworkClient();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType) {
        const TString requestKey = workloadId + "_" + ToString(networkHookType);
        auto node = CreateNode(
            workloadStatusRepository
            , workloadId
            , networkHookType
            , NStatusRepositoryTypes::EHookBackend::HTTP
            , requestHash
            , networkClient
            , expectedAnswer
            , false
        );
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);

        UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpRequestsWrongAnswerCounter(workloadId, networkHookType), 1);
        UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpHookConsecutiveFailuresCounter(workloadId, networkHookType), 1);
        UNIT_ASSERT_EQUAL(GetHttpHookStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::EHttpGetState::EHttpGetState_WRONG_ANSWER);

        UNIT_ASSERT_EQUAL(requestKey, ((TMyNetworkClient*)networkClient.Get())->RequestKey_);
        UNIT_ASSERT_EQUAL(requestHash, ((TMyNetworkClient*)networkClient.Get())->RequestHash_);
    };

    TestHttpHookAllTypes(test);
}

Y_UNIT_TEST(TestGetAndRemoveWithNonTimeoutError) {
    struct TMyNetworkClientFails : public TMockNetworkClient {
        TExpected<TString, TNetworkClientError> GetAndRemoveRequestResponse(const TString& requestKey, const TString& requestHash) override {
            RequestKey_ = requestKey;
            RequestHash_ = requestHash;
            return TNetworkClientError(
                ENetworkClientError::RequestHashMismatched
                , "bad hash"
            );
        }

        TString RequestKey_;
        TString RequestHash_;
    };

    const TString workloadId = "my_workload";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMyNetworkClientFails();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        const TString requestHash = "hash";
        const TString expectedAnswer = hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP ? "expected_answer": "";
        const TString requestKey = workloadId + "_" + ToString(networkHookType);
        auto node = CreateNode(
            workloadStatusRepository
            , workloadId
            , networkHookType
            , hookBackend
            , requestHash
            , networkClient
            , expectedAnswer
            , false
        );
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);

        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpRequestsErrorCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpHookConsecutiveFailuresCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(GetHttpHookStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::EHttpGetState::EHttpGetState_FAILURE);
        } else {
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetTcpCheckErrorCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetTcpCheckConsecutiveFailuresCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(GetTcpCheckStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::ETcpCheckState::ETcpCheckState_FAILURE);
        }

        UNIT_ASSERT_EQUAL(requestKey, ((TMyNetworkClientFails*)networkClient.Get())->RequestKey_);
        UNIT_ASSERT_EQUAL(requestHash, ((TMyNetworkClientFails*)networkClient.Get())->RequestHash_);
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestGetAndRemoveWithTimeoutError) {
    struct TMyNetworkClientFails : public TMockNetworkClient {
        TExpected<TString, TNetworkClientError> GetAndRemoveRequestResponse(const TString& requestKey, const TString& requestHash) override {
            RequestKey_ = requestKey;
            RequestHash_ = requestHash;
            return TNetworkClientError(
                ENetworkClientError::RequestTimeout
                , "timeout"
            );
        }

        TString RequestKey_;
        TString RequestHash_;
    };

    const TString workloadId = "my_workload";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMyNetworkClientFails();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        const TString requestHash = "hash";
        const TString expectedAnswer = hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP ? "expected_answer": "";
        const TString requestKey = workloadId + "_" + ToString(networkHookType);
        auto node = CreateNode(
            workloadStatusRepository
            , workloadId
            , networkHookType
            , hookBackend
            , requestHash
            , networkClient
            , expectedAnswer
            , false
        );
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);

        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpRequestsTimeoutCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetHttpHookConsecutiveFailuresCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(GetHttpHookStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::EHttpGetState::EHttpGetState_FAILURE);
        } else {
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetTcpCheckTimeoutCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(workloadStatusRepository->GetTcpCheckConsecutiveFailuresCounter(workloadId, networkHookType), 1);
            UNIT_ASSERT_EQUAL(GetTcpCheckStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::ETcpCheckState::ETcpCheckState_FAILURE);
        }

        UNIT_ASSERT_EQUAL(requestKey, ((TMyNetworkClientFails*)networkClient.Get())->RequestKey_);
        UNIT_ASSERT_EQUAL(requestHash, ((TMyNetworkClientFails*)networkClient.Get())->RequestHash_);
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(HttpAnyAnswerWithExpectedAnswer) {
    const TString workloadId = "my_workload";
    const TString requestHash = "hash";
    const TString expectedAnswer = "expected_answer";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMockNetworkClient();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            CreateNode(
                workloadStatusRepository
                , workloadId
                , networkHookType
                , NStatusRepositoryTypes::EHookBackend::HTTP
                , requestHash
                , networkClient
                , "Data"
                , true
            )
            , yexception
            , "AnyAnswer and ExpectedAnswer provided simultaneously"
        );
    };

    TestHttpHookAllTypes(test);
}

}

} // namespace NInfra::NPodAgent::NTestNetworkGetAndRemoveRequestResponseNode
