#include "network_get_request_state_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/network_client/mock_client.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/test_functions.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestNetworkGetRequestStateNode {

static TLogger logger({});

TNetworkGetRequestStateNodePtr CreateNode(
    TWorkloadStatusRepositoryPtr workloadStatusRepository
    , const TString& workloadId
    , NStatusRepositoryTypes::ENetworkHookType networkHookType
    , const TString& requestHash
    , TNetworkClientPtr networkClient
    , NStatusRepositoryTypes::EHookBackend hookBackend
) {
    return new TNetworkGetRequestStateNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , workloadStatusRepository
        , workloadId
        , networkHookType
        , requestHash
        , networkClient
        , hookBackend
    );
}

Y_UNIT_TEST_SUITE(NetworkGetRequestStateNodeSuite) {

Y_UNIT_TEST(TestGetState) {
    struct TMyNetworkClient : public TMockNetworkClient {
        TExpected<ERequestState, TNetworkClientError> GetRequestState(const TString& requestKey, const TString& requestHash) override {
            RequestKey_ = requestKey;
            RequestHash_ = requestHash;
            return INetworkClient::ERequestState::RUNNING;
        }

        TString RequestKey_;
        TString RequestHash_;
    };

    const TString workloadId = "my_workload";
    const TString requestHash = "hash";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMyNetworkClient();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        const TString requestKey = workloadId + "_" + ToString(networkHookType);
        auto node = CreateNode(
            workloadStatusRepository
            , workloadId
            , networkHookType
            , requestHash
            , networkClient
            , hookBackend
        );
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
        UNIT_ASSERT_EQUAL(ToString(INetworkClient::ERequestState::RUNNING), result.Success().Message);
        UNIT_ASSERT_EQUAL(requestKey, ((TMyNetworkClient*)networkClient.Get())->RequestKey_);
        UNIT_ASSERT_EQUAL(requestHash, ((TMyNetworkClient*)networkClient.Get())->RequestHash_);
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestGetStateError) {
    struct TMyNetworkClientFails : public TMockNetworkClient {
        TExpected<ERequestState, TNetworkClientError> GetRequestState(const TString& requestKey, const TString& requestHash) override {
            RequestKey_ = requestKey;
            RequestHash_ = requestHash;
            return TNetworkClientError(
                ENetworkClientError::RequestHashMismatched
                , "bad hash"
            );
        }

        TString RequestKey_;
        TString RequestHash_;
    };

    const TString workloadId = "my_workload";
    const TString requestHash = "hash";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMyNetworkClientFails();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        const TString requestKey = workloadId + "_" + ToString(networkHookType);
        auto node = CreateNode(
            workloadStatusRepository
            , workloadId
            , networkHookType
            , requestHash
            , networkClient
            , hookBackend
        );

        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            workloadStatusRepository->UpdateHttpHookState(workloadId, API::EHttpGetState::EHttpGetState_RUNNING, networkHookType);
        } else {
            workloadStatusRepository->UpdateTcpCheckState(workloadId, API::ETcpCheckState::ETcpCheckState_RUNNING, networkHookType);
        }
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
        UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "bad hash");

        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            UNIT_ASSERT_EQUAL(GetHttpHookStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::EHttpGetState::EHttpGetState_UNKNOWN);
        } else {
            UNIT_ASSERT_EQUAL(GetTcpCheckStatus(workloadStatusRepository->GetObjectStatus(workloadId), networkHookType).current().state(), API::ETcpCheckState::ETcpCheckState_UNKNOWN);
        }

        UNIT_ASSERT_EQUAL(requestKey, ((TMyNetworkClientFails*)networkClient.Get())->RequestKey_);
        UNIT_ASSERT_EQUAL(requestHash, ((TMyNetworkClientFails*)networkClient.Get())->RequestHash_);
    };

    TestNetworkHooksAllTypes(test);
}

}

} // namespace NInfra::NPodAgent::NTestNetworkGetRequestStateNode
