#include "network_hook_consecutive_tries_reached_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/test_functions.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestNetworkHookConsecutiveTriesReachedNode {

TNetworkHookConsecutiveTriesReachedNodePtr CreateNode(
    TWorkloadStatusRepositoryPtr statusRepository
    , const TString& id
    , NStatusRepositoryTypes::ENetworkHookType networkHookType
    , NStatusRepositoryTypes::EHookBackend hookBackend
    , bool countSuccesses
    , ui32 triesThreshold
) {
    return TNetworkHookConsecutiveTriesReachedNodePtr(new TNetworkHookConsecutiveTriesReachedNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , statusRepository
        , id
        , networkHookType
        , hookBackend
        , countSuccesses
        , triesThreshold
    ));
}

static TLogger logger({});
static TString WORKLOAD_ID = "my_workload";

Y_UNIT_TEST_SUITE(NetworkHookConsecutiveTriesReachedSuite) {

Y_UNIT_TEST(TestNetworkHookTriesNotReached) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(WORKLOAD_ID));

    for (int i = 0; i <= 1; ++i) {
        auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
            bool resultSuccess = (bool)i;
            switch (hookBackend) {
                case NStatusRepositoryTypes::EHookBackend::HTTP:
                    holder->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(WORKLOAD_ID, networkHookType, resultSuccess);
                    break;
                case NStatusRepositoryTypes::EHookBackend::TCP:
                    holder->UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(WORKLOAD_ID, networkHookType, resultSuccess);
                    break;
                case NStatusRepositoryTypes::EHookBackend::CONTAINER:
                case NStatusRepositoryTypes::EHookBackend::UNIX_SIGNAL:
                case NStatusRepositoryTypes::EHookBackend::NO_HOOK:
                    break;
            }

            auto node = CreateNode(holder, WORKLOAD_ID, networkHookType, hookBackend, resultSuccess, 3);

            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_C(result, result.Error().Message);
            UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
        };

        TestNetworkHooksAllTypes(test);
    }
}

Y_UNIT_TEST(TestNetworkHookTriesReached) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(WORKLOAD_ID));

    for (int i = 0; i <= 1; ++i) {
        auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
            bool resultSuccess = (bool)i;
            switch (hookBackend) {
                case NStatusRepositoryTypes::EHookBackend::HTTP:
                    holder->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(WORKLOAD_ID, networkHookType, resultSuccess);
                    holder->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(WORKLOAD_ID, networkHookType, resultSuccess);
                    holder->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(WORKLOAD_ID, networkHookType, resultSuccess);
                    break;
                case NStatusRepositoryTypes::EHookBackend::TCP:
                    holder->UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(WORKLOAD_ID, networkHookType, resultSuccess);
                    holder->UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(WORKLOAD_ID, networkHookType, resultSuccess);
                    holder->UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(WORKLOAD_ID, networkHookType, resultSuccess);
                    break;
                case NStatusRepositoryTypes::EHookBackend::CONTAINER:
                case NStatusRepositoryTypes::EHookBackend::UNIX_SIGNAL:
                case NStatusRepositoryTypes::EHookBackend::NO_HOOK:
                    break;
            }

            auto node = CreateNode(holder, WORKLOAD_ID, networkHookType, hookBackend, resultSuccess, 3);

            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_C(result, result.Error().Message);
            UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
        };

        TestNetworkHooksAllTypes(test);
    }
}

Y_UNIT_TEST(TestBadTriesThreshold) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(WORKLOAD_ID));

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            CreateNode(holder, WORKLOAD_ID, networkHookType, hookBackend, true, 0)
            , yexception
            , "triesThreshold should not be zero"
        );
    };

    TestNetworkHooksAllTypes(test);
}

}

} // namespace NInfra::NPodAgent::NTestNetworkHookConsecutiveTriesReachedNode
