#include "network_hook_start_time_expired_with_backoff_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/test_functions.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/generic/serialized_enum.h>

namespace NInfra::NPodAgent::NTestNetworkHookStartTimeExpiredWithBackoffNode {

TNetworkHookStartTimeExpiredWithBackoffNodePtr CreateNode(
    TWorkloadStatusRepositoryPtr statusRepository
    , const TString& id
    , NStatusRepositoryTypes::ENetworkHookType networkHookType
    , NStatusRepositoryTypes::EHookBackend hookBackend
    , TDuration restartPeriodScale
    , ui64 restartPeriodBackOff
    , TDuration maxRestartPeriod
    , TDuration minRestartPeriod
) {
    return TNetworkHookStartTimeExpiredWithBackoffNodePtr(
        new TNetworkHookStartTimeExpiredWithBackoffNode(
            TBasicTreeNodeDescriptor{1, "title"}
            , statusRepository
            , id
            , networkHookType
            , hookBackend
            , restartPeriodScale
            , restartPeriodBackOff
            , maxRestartPeriod
            , minRestartPeriod
        )
    );
}

static TLogger logger({});

Y_UNIT_TEST_SUITE(NetworkHookStartTimeExpiredWithBackoffSuite) {

Y_UNIT_TEST(TestNetworkHookStartTimeExpired) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            holder->UpdateHttpHookStartTime(id, TInstant::Seconds(10), networkHookType);
            holder->CaptureHttpHookStatus(id, networkHookType);
        } else {
            holder->UpdateTcpCheckStartTime(id, TInstant::Seconds(10), networkHookType);
            holder->CaptureTcpHookStatus(id, networkHookType);
        }

        TDuration duration = TDuration::Seconds(60);
        auto node = CreateNode(holder, id, networkHookType, hookBackend, TDuration::FromValue(0), 1, duration, duration);

        auto context = MockTickContext(logger);
        auto result = node->Tick(context);

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL_C(0, context->GetLongTickPeriodDuration(), context->GetLongTickPeriodDuration());
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestNetworkHookStartTimeNotExpired) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            holder->UpdateHttpHookStartTime(id, TInstant::Seconds(4000000000), networkHookType);
            holder->CaptureHttpHookStatus(id, networkHookType);
        } else {
            holder->UpdateTcpCheckStartTime(id, TInstant::Seconds(4000000000), networkHookType);
            holder->CaptureTcpHookStatus(id, networkHookType);
        }

        TDuration duration = TDuration::Seconds(60);
        auto node = CreateNode(holder, id, networkHookType, hookBackend, TDuration::FromValue(0), 1, duration, duration);

        auto context = MockTickContext(logger);
        auto result = node->Tick(context);

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestNetworkHookStartTimeNowNotExpired) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            holder->UpdateHttpHookStartTime(id, TInstant::Now(), networkHookType);
            holder->CaptureHttpHookStatus(id, networkHookType);
        } else {
            holder->UpdateTcpCheckStartTime(id, TInstant::Now(), networkHookType);
            holder->CaptureTcpHookStatus(id, networkHookType);
        }

        TDuration duration = TDuration::Seconds(60);
        auto node = CreateNode(holder, id, networkHookType, hookBackend, TDuration::FromValue(0), 1, duration, duration);

        auto context = MockTickContext(logger);
        auto result = node->Tick(context);

        UNIT_ASSERT(labs(60000 - (long) context->GetLongTickPeriodDuration()) < 1000);
        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestNetworkHookBackOff) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        bool hookSuccess = networkHookType != NStatusRepositoryTypes::ENetworkHookType::STOP && networkHookType != NStatusRepositoryTypes::ENetworkHookType::DESTROY;
        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            holder->UpdateHttpHookStartTime(id, TInstant::Now() - TDuration::Seconds(59), networkHookType);
            holder->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(id, networkHookType, hookSuccess);
            holder->CaptureHttpHookStatus(id, networkHookType);
        } else {
            holder->UpdateTcpCheckStartTime(id, TInstant::Now() - TDuration::Seconds(59), networkHookType);
            holder->UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(id, networkHookType, hookSuccess);
            holder->CaptureTcpHookStatus(id, networkHookType);
        }

        TDuration second = TDuration::Seconds(1);
        auto node = CreateNode(holder, id, networkHookType, hookBackend, second, 1000, second * 1000, second);

        auto context = MockTickContext(logger);
        auto result = node->Tick(context);

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            holder->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(id, networkHookType, hookSuccess);
        } else {
            holder->UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(id, networkHookType, hookSuccess);
        }

        context = MockTickContext(logger);
        result = node->Tick(context);

        UNIT_ASSERT(labs(1000000 - 59000 - (long) context->GetLongTickPeriodDuration()) < 1000);
        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestNetworkHookNoBackOff) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        bool hookSuccess = networkHookType != NStatusRepositoryTypes::ENetworkHookType::STOP && networkHookType != NStatusRepositoryTypes::ENetworkHookType::DESTROY;
        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            holder->UpdateHttpHookStartTime(id, TInstant::Now() - TDuration::Seconds(59), networkHookType);
            holder->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(id, networkHookType, !hookSuccess);
            holder->CaptureHttpHookStatus(id, networkHookType);
        } else {
            holder->UpdateTcpCheckStartTime(id, TInstant::Now() - TDuration::Seconds(59), networkHookType);
            holder->UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(id, networkHookType, !hookSuccess);
            holder->CaptureTcpHookStatus(id, networkHookType);
        }

        TDuration second = TDuration::Seconds(1);
        auto node = CreateNode(holder, id, networkHookType, hookBackend, second, 1000, second * 1000, second);

        auto context = MockTickContext(logger);
        auto result = node->Tick(context);

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

        context = MockTickContext(logger);
        result = node->Tick(context);

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestNetworkHookDoubleCheck) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        bool hookSuccess = networkHookType != NStatusRepositoryTypes::ENetworkHookType::STOP && networkHookType != NStatusRepositoryTypes::ENetworkHookType::DESTROY;
        if (hookBackend == NStatusRepositoryTypes::EHookBackend::HTTP) {
            holder->UpdateHttpHookStartTime(id, TInstant::Now() - TDuration::Seconds(59), networkHookType);
            holder->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(id, networkHookType, hookSuccess);
            holder->CaptureHttpHookStatus(id, networkHookType);
        } else {
            holder->UpdateTcpCheckStartTime(id, TInstant::Now() - TDuration::Seconds(59), networkHookType);
            holder->UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(id, networkHookType, hookSuccess);
            holder->CaptureTcpHookStatus(id, networkHookType);
        }

        TDuration second = TDuration::Seconds(1);
        auto node = CreateNode(holder, id, networkHookType, hookBackend, second, 1000, second * 1000, second);

        auto context = MockTickContext(logger);
        auto result = node->Tick(context);

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

        context = MockTickContext(logger);
        result = node->Tick(context);

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    };

    TestNetworkHooksAllTypes(test);
}

Y_UNIT_TEST(TestWithUndefinedWorkloadStatusRepository) {
    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType, NStatusRepositoryTypes::EHookBackend hookBackend) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            CreateNode(
                nullptr
                , "my_workload"
                , networkHookType
                , hookBackend
                , TDuration::Seconds(0)
                , 0
                , TDuration::Seconds(0)
                , TDuration::Seconds(0)
            )
            , yexception
            , "WorkloadStatusRepository not defined"
        );
    };

    TestNetworkHooksAllTypes(test);
}

}

} // namespace NInfra::NPodAgent::NTestNetworkHookStartTimeExpiredWithBackoffNode
