#include "network_remove_request_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/network_client/mock_client.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/test_functions.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestNetworkRemoveRequestNode {

static TLogger logger({});

TNetworkRemoveRequestNodePtr CreateNode(
    TWorkloadStatusRepositoryPtr workloadStatusRepository
    , const TString& workloadId
    , NStatusRepositoryTypes::ENetworkHookType networkHookType
    , TNetworkClientPtr networkClient
) {
    return new TNetworkRemoveRequestNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , workloadStatusRepository
        , workloadId
        , networkHookType
        , networkClient
    );
}

Y_UNIT_TEST_SUITE(NetworkRemoveRequestNodeSuite) {

Y_UNIT_TEST(TestRemoveRequest) {
    struct TMyNetworkClient : public TMockNetworkClient {
        TExpected<void, TNetworkClientError> RemoveRequest(const TString& requestKey) override {
            RequestKey_ = requestKey;
            return TExpected<void, TNetworkClientError>::DefaultSuccess();
        }

        TString RequestKey_;
    };

    const TString workloadId = "my_workload";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMyNetworkClient();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType) {
        const TString requestKey = workloadId + "_" + ToString(networkHookType);
        auto node = CreateNode(
            workloadStatusRepository
            , workloadId
            , networkHookType
            , networkClient
        );
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
        UNIT_ASSERT_EQUAL(requestKey, ((TMyNetworkClient*)networkClient.Get())->RequestKey_);
    };

    TestHttpHookAllTypes(test);
    TestTcpHookAllTypes(test);
}

Y_UNIT_TEST(TestRemoveRequestWithError) {
    struct TMyNetworkClientFails : public TMockNetworkClient {
        TExpected<void, TNetworkClientError> RemoveRequest(const TString& requestKey) override {
            RequestKey_ = requestKey;
            return TNetworkClientError(
                ENetworkClientError::Unspecified
                , "bad request"
            );
        }

        TString RequestKey_;
    };

    const TString workloadId = "my_workload";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMyNetworkClientFails();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType) {
        const TString requestKey = workloadId + "_" + ToString(networkHookType);
        auto node = CreateNode(
            workloadStatusRepository
            , workloadId
            , networkHookType
            , networkClient
        );
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(!result, result.Success().Status);
        UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "bad request");
        UNIT_ASSERT_EQUAL(requestKey, ((TMyNetworkClientFails*)networkClient.Get())->RequestKey_);
    };

    TestHttpHookAllTypes(test);
    TestTcpHookAllTypes(test);
}

Y_UNIT_TEST(TestRemoveRequestDoesNotExist) {
    struct TMyNetworkClient : public TMockNetworkClient {
        TExpected<void, TNetworkClientError> RemoveRequest(const TString& requestKey) override {
            RequestKey_ = requestKey;
            return TNetworkClientError(
                ENetworkClientError::RequestDoesNotExist
                , "Does not exist"
            );
        }

        TString RequestKey_;
    };

    const TString workloadId = "my_workload";

    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TNetworkClientPtr networkClient = new TMyNetworkClient();

    auto test = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType) {
        const TString requestKey = workloadId + "_" + ToString(networkHookType);
        auto node = CreateNode(
            workloadStatusRepository
            , workloadId
            , networkHookType
            , networkClient
        );
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
        UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "Already removed");
        UNIT_ASSERT_EQUAL(requestKey, ((TMyNetworkClient*)networkClient.Get())->RequestKey_);
    };

    TestHttpHookAllTypes(test);
    TestTcpHookAllTypes(test);
}

}

} // namespace NInfra::NPodAgent::NTestNetworkRemoveRequestNode
