#include "porto_add_properties_to_container_fail_reason_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>
#include <infra/pod_agent/libs/porto_client/test_functions.h>
#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NPortoTest::NTestPortoAddPropertiesToContainerFailReasonNode {

using namespace NPortoTest;

static TLogger logger({});
static const TVector<EPortoContainerProperty> PROPERTIES = {
    EPortoContainerProperty::ExitCode
    , EPortoContainerProperty::StdErr
    , EPortoContainerProperty::StdOut
};

TPortoAddPropertiesToContainerFailReasonNodePtr CreateNode(
    const TStatusRepositoryCommonPtr statusRepositoryCommon
    , const NStatusRepositoryTypes::TContainerDescription& container
    , const TPortoContainerName containerName
    , const TAsyncPortoClientPtr porto
    , const TVector<EPortoContainerProperty>& property
) {
    return new TPortoAddPropertiesToContainerFailReasonNode(
        {1, "title"}
        , porto
        , containerName
        , statusRepositoryCommon
        , container
        , property
    );
}

NStatusRepositoryTypes::TContainerDescription CreateContainerDescription(const TString& id) {
    return NStatusRepositoryTypes::TContainerDescription(id, NStatusRepositoryTypes::WORKLOAD, NStatusRepositoryTypes::TContainerDescription::READINESS);
}

TPortoAddPropertiesToContainerFailReasonNodePtr PrepareTestObjects(
    TWorkloadStatusRepositoryPtr holder,
    TAsyncPortoClientPtr porto,
    const TString& containerId,
    const NStatusRepositoryTypes::TContainerDescription& container
) {
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(containerId));
    return CreateNode(
        holder
        , container
        , TPortoContainerName(containerId)
        , porto
        , PROPERTIES
    );
}


Y_UNIT_TEST_SUITE(PortoAddPropertiesToContainerFailReasonNodeSuite) {

Y_UNIT_TEST(TestPortoAddPropertyToFailReason) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();

    struct TPortoSuccessfulClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& name
            , const TVector<EPortoContainerProperty>& variables
        ) override {
            UNIT_ASSERT_EQUAL(name, Name_);
            UNIT_ASSERT_EQUAL(variables, Properties_);

            TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;

            for (auto& currentVariable : variables) {
                result[name[0]][currentVariable] = CreateTPortoGetResponse(ToString(currentVariable) + "_value", 0, "");
            }

            return result;
        }

        const TVector<TPortoContainerName> Name_ = TVector<TPortoContainerName>({TPortoContainerName("my_workload")});
        const TVector<EPortoContainerProperty> Properties_ = PROPERTIES;
    };

    TPortoClientPtr myPorto = new TPortoSuccessfulClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString id = "my_workload";
    auto container = CreateContainerDescription(id);
    auto node = PrepareTestObjects(holder, porto, id, container);

    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL_C(result.Success().Status, ENodeStatus::SUCCESS, ToString(result.Success().Status));

    auto failReason = holder->GetContainerFailReason(container);
    TString message = "exit_code = 'exit_code_value', stderr = 'stderr_value', stdout = 'stdout_value'";

    UNIT_ASSERT_EQUAL_C(message, failReason, failReason);
}

Y_UNIT_TEST(TestFailedPorto) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();

    struct TPortoFailingClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& name
            , const TVector<EPortoContainerProperty>& variables
        ) override {
            UNIT_ASSERT_EQUAL(name, Name_);
            UNIT_ASSERT_EQUAL(variables, Properties_);

            return TPortoError{EPortoError::Unknown, "error message", "NO"};
        }

        const TVector<TPortoContainerName> Name_ = TVector<TPortoContainerName>({TPortoContainerName("my_workload")});
        const TVector<EPortoContainerProperty> Properties_ = PROPERTIES;
    };

    TPortoClientPtr myPorto = new TPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString id = "my_workload";
    auto container = CreateContainerDescription(id);
    auto node = PrepareTestObjects(holder, porto, id, container);

    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "error message");
}

Y_UNIT_TEST(TestFailedPortoProperty) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();

    struct TPortoFailingClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& name
            , const TVector<EPortoContainerProperty>& variables
        ) override {
            UNIT_ASSERT_EQUAL(name, Name_);
            UNIT_ASSERT_EQUAL(variables, Properties_);

            TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;

            for (auto& currentVariable : variables) {
                result[name[0]][currentVariable] = CreateTPortoGetResponse(ToString(currentVariable) + "_value", 1, "bad property");
            }

            return result;
        }

        const TVector<TPortoContainerName> Name_ = TVector<TPortoContainerName>({TPortoContainerName("my_workload")});
        const TVector<EPortoContainerProperty> Properties_ = PROPERTIES;
    };

    TPortoClientPtr myPorto = new TPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString id = "my_workload";
    auto container = CreateContainerDescription(id);
    auto node = PrepareTestObjects(holder, porto, id, container);

    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "bad property");
}

Y_UNIT_TEST(TestFailedPortoMissedProperty) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();

    struct TPortoFailingClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
                const TVector<TPortoContainerName>& name
                , const TVector<EPortoContainerProperty>& variables
        ) override {
            UNIT_ASSERT_EQUAL(name, Name_);
            UNIT_ASSERT_EQUAL(variables, Properties_);

            TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;

            for (auto& currentVariable : variables) {
                if (currentVariable == variables[0]) {
                    continue;
                }
                result[name[0]][currentVariable] = CreateTPortoGetResponse(ToString(currentVariable) + "_value", 0, "");
            }

            return result;
        }

        const TVector<TPortoContainerName> Name_ = TVector<TPortoContainerName>({TPortoContainerName("my_workload")});
        const TVector<EPortoContainerProperty> Properties_ = PROPERTIES;
    };

    TPortoClientPtr myPorto = new TPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString id = "my_workload";
    auto container = CreateContainerDescription(id);
    auto node = PrepareTestObjects(holder, porto, id, container);

    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "properties incomplete");
}

Y_UNIT_TEST(TestFailedPortoMissedContainer) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();

    struct TPortoFailingClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& name
            , const TVector<EPortoContainerProperty>& variables
        ) override {
            UNIT_ASSERT_EQUAL(name, Name_);
            UNIT_ASSERT_EQUAL(variables, Properties_);

            TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;

            for (auto& currentVariable : variables) {
                result[TPortoContainerName("random_name")][currentVariable] = CreateTPortoGetResponse(ToString(currentVariable) + "_value", 0, "");
            }

            return result;
        }

        const TVector<TPortoContainerName> Name_ = TVector<TPortoContainerName>({TPortoContainerName("my_workload")});
        const TVector<EPortoContainerProperty> Properties_ = PROPERTIES;
    };

    TPortoClientPtr myPorto = new TPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString id = "my_workload";
    auto container = CreateContainerDescription(id);
    auto node = PrepareTestObjects(holder, porto, id, container);

    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "porto get call doesn't contain");
}

}

} // namespace NInfra::NPodAgent::NTestPortoAddPropertiesToContainerFailReasonNode
