#include "porto_add_properties_to_object_failed_message_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/layer_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>
#include <infra/pod_agent/libs/porto_client/test_functions.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NPortoTest::NTestPortoAddPropertyToObjectFailedMessageNode {

using namespace NPortoTest;

static TLogger logger({});

Y_UNIT_TEST_SUITE(PortoAddPropertyToObjectFailedMessageNodeSuite) {

TPortoAddPropertiesToObjectFailedMessageNodePtr CreateNode(
        const TStatusRepositoryCommonPtr statusRepositoryCommon
        , const TString& objectIdOrHash
        , const TPortoContainerName containerName
        , const TAsyncPortoClientPtr porto
        , const TVector<EPortoContainerProperty>& property
) {
    return new TPortoAddPropertiesToObjectFailedMessageNode(
        {1, "title"}
        , porto
        , containerName
        , statusRepositoryCommon
        , objectIdOrHash
        , property
    );
}

Y_UNIT_TEST(TestPortoAddPropertyToFailedMessage) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();

    struct TPortoSuccessfulClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& name
            , const TVector<EPortoContainerProperty>& variable
        ) override {
            UNIT_ASSERT_EQUAL(name, Name_);
            UNIT_ASSERT_EQUAL(variable, Variables_);

            TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;

            for (auto& currentVariable : variable) {
                result[name[0]][currentVariable] = CreateTPortoGetResponse(ToString(currentVariable) + "_value", 0, "");
            }

            return result;
        }

        const TVector<TPortoContainerName> Name_ = TVector<TPortoContainerName>({TPortoContainerName("my_container")});
        const TVector<EPortoContainerProperty> Variables_ = TVector<EPortoContainerProperty>({
            EPortoContainerProperty::ExitCode
            , EPortoContainerProperty::StdErr
            , EPortoContainerProperty::StdOut
        });
    };

    TPortoClientPtr myPorto = new TPortoSuccessfulClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString id = "my_layer";
    const TString hash = "my_layer_download_hash";
    TPortoContainerName containerName("my_container");
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, hash));

    auto node = CreateNode(
        holder
        , hash
        , containerName
        , porto
        , TVector<EPortoContainerProperty>({
            EPortoContainerProperty::ExitCode
            , EPortoContainerProperty::StdErr
            , EPortoContainerProperty::StdOut
        })
    );
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL_C(result.Success().Status, ENodeStatus::SUCCESS, ToString(result.Success().Status));

    auto status = holder->GetObjectStatus(id);
    TString message = "exit_code = 'exit_code_value'\nstderr = 'stderr_value'\nstdout = 'stdout_value'";

    UNIT_ASSERT_EQUAL_C(message, status.failed().message(), status.failed().message());
    UNIT_ASSERT_EQUAL_C(1, status.fail_counter(), status.fail_counter());
}

Y_UNIT_TEST(TestFailedPorto) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();

    struct TPortoFailingClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& name
            , const TVector<EPortoContainerProperty>& variable
        ) override {
            UNIT_ASSERT_EQUAL(name, Name_);
            UNIT_ASSERT_EQUAL(variable, Variables_);

            return TPortoError{EPortoError::Unknown, "error message", "NO"};
        }

        const TVector<TPortoContainerName> Name_ = TVector<TPortoContainerName>({TPortoContainerName("my_container")});
        const TVector<EPortoContainerProperty> Variables_ = TVector<EPortoContainerProperty>({
            EPortoContainerProperty::ExitCode
            , EPortoContainerProperty::StdErr
            , EPortoContainerProperty::StdOut
        });
    };

    TPortoClientPtr myPorto = new TPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString id = "my_layer";
    const TString hash = "my_layer_download_hash";
    TPortoContainerName containerName("my_container");
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, hash));

    auto node = CreateNode(
        holder
        , hash
        , containerName
        , porto
        , TVector<EPortoContainerProperty>({
            EPortoContainerProperty::ExitCode
            , EPortoContainerProperty::StdErr, EPortoContainerProperty::StdOut
        })
    );
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "error message");
}

Y_UNIT_TEST(TestFailedPortoProperty) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();

    struct TPortoFailingClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& name
            , const TVector<EPortoContainerProperty>& variable
        ) override {
            UNIT_ASSERT_EQUAL(name, Name_);
            UNIT_ASSERT_EQUAL(variable, Variables_);

            TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;

            for (auto& currentVariable : variable) {
                result[name[0]][currentVariable] = CreateTPortoGetResponse(ToString(currentVariable) + "_value", 1, "bad property");
            }

            return result;
        }

        const TVector<TPortoContainerName> Name_ = TVector<TPortoContainerName>({TPortoContainerName("my_container")});
        const TVector<EPortoContainerProperty> Variables_ = TVector<EPortoContainerProperty>({
            EPortoContainerProperty::ExitCode
            , EPortoContainerProperty::StdErr
            , EPortoContainerProperty::StdOut
        });
    };

    TPortoClientPtr myPorto = new TPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString id = "my_layer";
    const TString hash = "my_layer_download_hash";
    TPortoContainerName containerName("my_container");
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, hash));

    auto node = CreateNode(
        holder
        , hash
        , containerName
        , porto
        , TVector<EPortoContainerProperty>({
            EPortoContainerProperty::ExitCode
            , EPortoContainerProperty::StdErr
            , EPortoContainerProperty::StdOut
        })
    );
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "bad property");
}

Y_UNIT_TEST(TestFailedPortoMissedProperty) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();

    struct TPortoFailingClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& name
            , const TVector<EPortoContainerProperty>& variable
        ) override {
            UNIT_ASSERT_EQUAL(name, Name_);
            UNIT_ASSERT_EQUAL(variable, Variables_);

            TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;

            for (auto& currentVariable : variable) {
                if (currentVariable == variable[0]) {
                    continue;
                }
                result[name[0]][currentVariable] = CreateTPortoGetResponse(ToString(currentVariable) + "_value", 0, "");
            }

            return result;
        }

        const TVector<TPortoContainerName> Name_ = TVector<TPortoContainerName>({TPortoContainerName("my_container")});
        const TVector<EPortoContainerProperty> Variables_ = TVector<EPortoContainerProperty>({
            EPortoContainerProperty::ExitCode
            , EPortoContainerProperty::StdErr
            , EPortoContainerProperty::StdOut
        });
    };

    TPortoClientPtr myPorto = new TPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString id = "my_layer";
    const TString hash = "my_layer_download_hash";
    TPortoContainerName containerName("my_container");
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, hash));

    auto node = CreateNode(
        holder
        , hash
        , containerName
        , porto
        , TVector<EPortoContainerProperty>({
            EPortoContainerProperty::ExitCode
            , EPortoContainerProperty::StdErr
            , EPortoContainerProperty::StdOut
        })
    );
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "properties incomplete");
}

Y_UNIT_TEST(TestFailedPortoMissedContainer) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();

    struct TPortoFailingClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& name
            , const TVector<EPortoContainerProperty>& variable
        ) override {
            UNIT_ASSERT_EQUAL(name, Name_);
            UNIT_ASSERT_EQUAL(variable, Variables_);

            TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;

            for (auto& currentVariable : variable) {
                result[TPortoContainerName("random_name")][currentVariable] = CreateTPortoGetResponse(ToString(currentVariable) + "_value", 0, "");
            }

            return result;
        }

        const TVector<TPortoContainerName> Name_ = TVector<TPortoContainerName>({TPortoContainerName("my_container")});
        const TVector<EPortoContainerProperty> Variables_ = TVector<EPortoContainerProperty>({
            EPortoContainerProperty::ExitCode
            , EPortoContainerProperty::StdErr
            , EPortoContainerProperty::StdOut
        });
    };

    TPortoClientPtr myPorto = new TPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString id = "my_layer";
    const TString hash = "my_layer_download_hash";
    TPortoContainerName containerName("my_container");
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, hash));

    auto node = CreateNode(
        holder
        , hash
        , containerName
        , porto
        , TVector<EPortoContainerProperty>({
            EPortoContainerProperty::ExitCode
            , EPortoContainerProperty::StdErr
            , EPortoContainerProperty::StdOut
        })
    );
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "porto get call doesn't contain");
}

}

} // namespace NInfra::NPodAgent::NTestPortoAddPropertyToObjectFailedMessageNode
