#include "porto_clean_linked_volumes_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>
#include <infra/pod_agent/libs/porto_client/test_functions.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestPortoCleanLinkedVolumesNode {

Y_UNIT_TEST_SUITE(PortoCleanLinkedVolumesNodeSuite) {

static TLogger logger({});

TPortoCleanLinkedVolumesNodePtr CreateNode(
    TPortoClientPtr porto
    , const TString& path
    , const TPortoContainerName& container
) {
    return new TPortoCleanLinkedVolumesNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , new TAsyncPortoClient(porto, new TFakeThreadPool())
        , path
        , container
    );
}

Y_UNIT_TEST(TestRemoveLinkedVolumes) {
    auto test = [=](const TSet<TString>& volumes, size_t badVolumes) {
        struct TMyPortoClient : public TMockPortoClient {
            TMyPortoClient(const TSet<TString>& volumes, const TPortoContainerName& container)
                : LinkedVolumes_(volumes)
                , Container_(container)
            {}

            TExpected<TVector<TString>, TPortoError> ListVolumesPaths(const TString& /* path */, const TPortoContainerName& /* container */) override {
                ++Calls_;

                TVector<TString> result;
                for (auto& name : LinkedVolumes_) {
                    result.push_back(name);
                }

                return result;
            }

            TExpected<void, TPortoError> UnlinkVolume(const TString& name, const TPortoContainerName& container, const TString& path, bool strict) override {
                ++Calls_;
                LinkedVolumes_.erase(name);

                UNIT_ASSERT_EQUAL(name, path);
                UNIT_ASSERT(!strict);
                UNIT_ASSERT_EQUAL(container, Container_);

                return TExpected<void, TPortoError>::DefaultSuccess();
            }

            size_t GetCalls() const {
                return Calls_;
            }

            size_t GetSize() const {
                return LinkedVolumes_.size();
            }

            size_t Calls_ = 0;
            TSet<TString> LinkedVolumes_;
            const TPortoContainerName Container_;
        };

        TPortoContainerName container = TPortoContainerName::NoEscape("container");

        TPortoClientPtr porto = new TMyPortoClient(volumes, container);

        auto node = CreateNode(porto, "/root_path", container);
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_EQUAL_C(ENodeStatus::SUCCESS, result.Success().Status, result.Success().Message);
        UNIT_ASSERT_EQUAL(1 + badVolumes, ((TMyPortoClient*)porto.Get())->GetCalls());
        UNIT_ASSERT_EQUAL(volumes.size() - badVolumes, ((TMyPortoClient*)porto.Get())->GetSize());
    };

    TSet<TString> testResponse;
    testResponse.insert("/root_path/some/other/path");
    testResponse.insert("/root_path/short_one");
    testResponse.insert("/root_path/");

    // test unlink all volumes
    test(testResponse, testResponse.size());

    testResponse.clear();
    testResponse.insert("/some_other_path");
    testResponse.insert("/some_other_path/with/a/lot/of/levels");
    testResponse.insert("/yet_another_path/root_path");
    testResponse.insert("/yet_another_path/root_path/new_folder");
    testResponse.insert("/root_path_fake");

    // test volumes with paths not in root_path
    test(testResponse, 0);
}

}

} // namespace NInfra::NPodAgent::NTestPortoCleanLinkedVolumesNode
