#include "porto_container_exists_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoContainerExistsNode {

Y_UNIT_TEST_SUITE(PortoContainerExistsNodeSuite) {

static TLogger logger({});

TPortoContainerExistsNodePtr PortoCreator(TAsyncPortoClientPtr porto, const TPortoContainerName& name) {
    return new TPortoContainerExistsNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , name
    );
}

Y_UNIT_TEST(TestContainerExists) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<int, TPortoError> IsContainerExists(const TPortoContainerName& /*mask*/) override {
            ++Calls_;
            return 1;
        }

        size_t Calls_ = 0;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};

    auto creator = PortoCreator(porto, name);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
}

Y_UNIT_TEST(TestContainerNotExists) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<int, TPortoError> IsContainerExists(const TPortoContainerName& /*mask*/) override {
            ++Calls_;
            return 0;
        }

        size_t Calls_ = 0;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_2"};

    auto creator = PortoCreator(porto, name);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
}

Y_UNIT_TEST(TestContainerExistsError) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<int, TPortoError> IsContainerExists(const TPortoContainerName& /*mask*/) override {
            ++Calls_;
            return TPortoError{EPortoError::Unknown, "List", "MyCoolMessage"};
        }

        size_t Calls_ = 0;
    };

    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"some_name"};

    auto creator = PortoCreator(porto, name);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(false, (bool)result);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "MyCoolMessage");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoContainerExistsNode
