#include "porto_container_private_matches_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoContainerPrivateMatchesNode {

Y_UNIT_TEST_SUITE(TPortoContainerPrivateMatchesNodeSuite) {

static TLogger logger({});

TPortoContainerPrivateMatchesNodePtr CreateNode(
    TAsyncPortoClientPtr porto
    , const TPortoContainerName& name
    , const TString& treeHash
    , EContainerPrivate value
) {
    return new TPortoContainerPrivateMatchesNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , name
        , treeHash
        , value
    );
}

Y_UNIT_TEST(TestPrivateMatches) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;
            Property_ = property;

            return TString("ready:tree_hash");
        }

        TPortoContainerName Name_ = TString("");
        EPortoContainerProperty Property_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};

    auto creator = CreateNode(porto, name, "tree_hash", CP_READY);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::Private, ((TMyPortoClient*)myPorto.Get())->Property_);
}

Y_UNIT_TEST(TestPrivateNotMatches) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;
            Property_ = property;

            return TString("other_value:tree_hash");
        }

        TPortoContainerName Name_ = TString("");
        EPortoContainerProperty Property_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};

    auto creator = CreateNode(porto, name, "tree_hash", CP_READY);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::Private, ((TMyPortoClient*)myPorto.Get())->Property_);
}

Y_UNIT_TEST(TestContainerDoesNotExist) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;
            Property_ = property;

            TPortoError error;
            error.Code = EPortoError::ContainerDoesNotExist;

            return error;
        }

        TPortoContainerName Name_ = TString("");
        EPortoContainerProperty Property_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};

    auto creator = CreateNode(porto, name, "tree_hash", CP_READY);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::Private, ((TMyPortoClient*)myPorto.Get())->Property_);
}

Y_UNIT_TEST(TestPrivateWrongHash) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;
            Property_ = property;

            return TString("value:other_tree_hash");
        }

        TPortoContainerName Name_ = TString("");
        EPortoContainerProperty Property_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};

    auto creator = CreateNode(porto, name, "tree_hash", CP_READY);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::Private, ((TMyPortoClient*)myPorto.Get())->Property_);
}

Y_UNIT_TEST(TestPrivateWrongFormat) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;
            Property_ = property;

            return TString("value");
        }

        TPortoContainerName Name_ = TString("");
        EPortoContainerProperty Property_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};

    auto creator = CreateNode(porto, name, "tree_hash", CP_READY);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::Private, ((TMyPortoClient*)myPorto.Get())->Property_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoContainerPrivateMatchesNode
