#include "porto_container_property_matches_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoContainerPropertyMatchesNode {

Y_UNIT_TEST_SUITE(PortoContainerPropertyMatchesNodeSuite) {

static TLogger logger({});

TPortoContainerPropertyMatchesNodePtr CreateNode(
    TAsyncPortoClientPtr porto
    , const TPortoContainerName& name
    , EPortoContainerProperty property
    , const TString& value
) {
    return new TPortoContainerPropertyMatchesNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , name
        , property
        , value
    );
}

Y_UNIT_TEST(TestPropertyMatches) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;
            Property_ = property;

            return TString("value");
        }

        TPortoContainerName Name_ = TString("");
        EPortoContainerProperty Property_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TPortoContainerName name = {"container_0"};
    EPortoContainerProperty property = EPortoContainerProperty::Cwd;

    auto creator = CreateNode(porto, name, property, "value");
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
    UNIT_ASSERT_EQUAL(property, ((TMyPortoClient*)myPorto.Get())->Property_);
}

Y_UNIT_TEST(TestPropertyNotMatches) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;
            Property_ = property;

            return TString("other value");
        }

        TPortoContainerName Name_ = TString("");
        EPortoContainerProperty Property_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TPortoContainerName name = {"container_0"};
    EPortoContainerProperty property = EPortoContainerProperty::Cwd;

    auto creator = CreateNode(porto, name, property, "value");
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
    UNIT_ASSERT_EQUAL(property, ((TMyPortoClient*)myPorto.Get())->Property_);
}

Y_UNIT_TEST(TestStreamPropertyOffset) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetStdout(const TPortoContainerName& name, int offset, int length, int /*flags*/) override {
            Name_ = name;
            Offset_ = offset;
            Length_ = length;

            return TString("value");
        }

        TExpected<TString, TPortoError> GetStderr(const TPortoContainerName& name, int offset, int length, int flags) override {
            return GetStdout(name, offset, length, flags);
        }

        TPortoContainerName Name_ = TString("");
        int Offset_;
        int Length_;
    };

    for (EPortoContainerProperty property : {EPortoContainerProperty::StdOut, EPortoContainerProperty::StdErr}) {
        TPortoClientPtr myPorto = new TMyPortoClient();
        TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
        TPortoContainerName name = {"container_0"};

        auto creator = CreateNode(porto, name, property, "value");
        auto result = creator->Tick(MockTickContext(logger));
        TTickResult expected(ENodeStatus::SUCCESS);

        UNIT_ASSERT_EQUAL(expected, result);
        UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
        UNIT_ASSERT_EQUAL(-1, ((TMyPortoClient*)myPorto.Get())->Offset_);
        UNIT_ASSERT_EQUAL(TPortoContainerPropertyMatchesNode::STREAM_LENGTH, ((TMyPortoClient*)myPorto.Get())->Length_);
    }
}

}

} // namespace NInfra::NPodAgent::NTestPortoContainerPropertyMatchesNode
