#pragma once

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/common.h>
#include <infra/pod_agent/libs/behaviour/bt/nodes/base/porto_basic_node.h>

namespace NInfra::NPodAgent {

class TPortoContainerTimeExpiredNode;
using TPortoContainerTimeExpiredNodePtr = TSimpleSharedPtr<TPortoContainerTimeExpiredNode>;

class TPortoContainerTimeExpiredNode : public TPortoBasicNode<TExpected<TString, TPortoError>> {
public:
    enum class ETimeType {
        UNIX_TIMESTAMP    /* "unix_timestamp" */,
        ABSOLUTE_VALUE     /* "absolute_value" */,
    };

public:
    TPortoContainerTimeExpiredNode(
        const TBasicTreeNodeDescriptor& descriptor
        , TAsyncPortoClientPtr porto
        , const TPortoContainerName& name
        , TDuration expireTime
        , EPortoContainerProperty initialTime
        , ETimeType timeType
    )
        : TPortoBasicNode(descriptor, porto)
        , Name_(name)
        // Round expireTime to seconds because porto returns time in seconds
        , ExpireDuration_(RoundToSeconds(expireTime))
        , InitialTime_(initialTime)
        , TimeType_(timeType)
    {
        switch (TimeType_) {
            case ETimeType::UNIX_TIMESTAMP:
                Y_ENSURE(
                    InitialTime_ == EPortoContainerProperty::StartTimeRaw || InitialTime_ == EPortoContainerProperty::CreationTimeRaw
                    , "With time type '" << ToString(TimeType_)
                        << "' you can only use porto properties: 'start_time', 'creation_time'. But used '"
                        << ToString(InitialTime_) << "'"
                );
                break;
            case ETimeType::ABSOLUTE_VALUE:
                Y_ENSURE(
                    InitialTime_ == EPortoContainerProperty::Time
                    , "With time type '" << ToString(timeType)
                        << "' you can only use porto property 'time'. But used '"
                        << ToString(InitialTime_) << "'"
                );
                break;
        }
    }

    virtual ENodeType GetType() const override final;

private:
    virtual TExpected<TFuture<TExpected<TString, TPortoError>>, TTickResult> PortoCall(TTickContextPtr context) override final;
    virtual TTickResult ProcessPortoResult(TTickContextPtr context, TExpected<TString, TPortoError>& result) override final;

    TTickResult ProcessUnixTimestamp(TExpected<TString, TPortoError>& result);
    TTickResult ProcessAbsoluteValue(TExpected<TString, TPortoError>& result);

public:
    static constexpr const ENodeType NODE_TYPE = ENodeType::PORTO_CONTAINER_TIME_EXPIRED;

private:
    const TPortoContainerName Name_;
    const TDuration ExpireDuration_;
    const EPortoContainerProperty InitialTime_;
    const ETimeType TimeType_;
};

} // namespace NInfra::NPodAgent
