#include "porto_container_time_expired_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoContainerTimeExpiredNode {

Y_UNIT_TEST_SUITE(PortoContainerTimeExpiredNodeSuite) {

static TLogger logger({});

TPortoContainerTimeExpiredNodePtr CreateNode(
    TAsyncPortoClientPtr porto
    , const TPortoContainerName& name
    , TDuration duration
    , EPortoContainerProperty initialTime
    , TPortoContainerTimeExpiredNode::ETimeType timeType
) {
    return TPortoContainerTimeExpiredNodePtr(
        new TPortoContainerTimeExpiredNode(
            TBasicTreeNodeDescriptor{1, "title"}
            , porto
            , name
            , duration
            , initialTime
            , timeType
        )
    );
}

Y_UNIT_TEST(TestIncorrectInitialTimeArgument) {
    TPortoClientPtr myPorto = new TMockPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TPortoContainerName name = {"container_0"};
    TDuration duration = TDuration::Seconds(60);

    {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            CreateNode(porto, name, duration, EPortoContainerProperty::State, TPortoContainerTimeExpiredNode::ETimeType::UNIX_TIMESTAMP)
            , yexception
            , "With time type 'unix_timestamp' you can only use porto properties: 'start_time', 'creation_time'. But used 'state'"
        );
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            CreateNode(porto, name, duration, EPortoContainerProperty::State, TPortoContainerTimeExpiredNode::ETimeType::ABSOLUTE_VALUE)
            , yexception
            , "With time type 'absolute_value' you can only use porto property 'time'. But used 'state'"
        );
    }

    {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            CreateNode(porto, name, duration, EPortoContainerProperty::Time, TPortoContainerTimeExpiredNode::ETimeType::UNIX_TIMESTAMP)
            , yexception
            , "With time type 'unix_timestamp' you can only use porto properties: 'start_time', 'creation_time'. But used 'time'"
        );
    }

    {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            CreateNode(porto, name, duration, EPortoContainerProperty::StartTimeRaw, TPortoContainerTimeExpiredNode::ETimeType::ABSOLUTE_VALUE)
            , yexception
            , "With time type 'absolute_value' you can only use porto property 'time'. But used 'start_time[raw]'"
        );
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            CreateNode(porto, name, duration, EPortoContainerProperty::CreationTimeRaw, TPortoContainerTimeExpiredNode::ETimeType::ABSOLUTE_VALUE)
            , yexception
            , "With time type 'absolute_value' you can only use porto property 'time'. But used 'creation_time[raw]'"
        );
    }
}

Y_UNIT_TEST(TestStartTimeExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::StartTimeRaw) {
                return TString("10");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};
    TDuration duration = TDuration::Seconds(60);

    auto node = CreateNode(porto, name, duration, EPortoContainerProperty::StartTimeRaw, TPortoContainerTimeExpiredNode::ETimeType::UNIX_TIMESTAMP);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestCreationTimeExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::CreationTimeRaw) {
                return TString("10");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};
    TDuration duration = TDuration::Seconds(60);

    auto node = CreateNode(porto, name, duration, EPortoContainerProperty::CreationTimeRaw, TPortoContainerTimeExpiredNode::ETimeType::UNIX_TIMESTAMP);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestExecutionTimeExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::Time) {
                return TString("65");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};
    TDuration duration = TDuration::Seconds(60);

    auto node = CreateNode(porto, name, duration, EPortoContainerProperty::Time, TPortoContainerTimeExpiredNode::ETimeType::ABSOLUTE_VALUE);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestExecutionTimeExpiredWithZeroTimeAndDuration) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::Time) {
                return TString("0");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};
    TDuration duration = TDuration::Zero();

    auto node = CreateNode(porto, name, duration, EPortoContainerProperty::Time, TPortoContainerTimeExpiredNode::ETimeType::ABSOLUTE_VALUE);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestStartTimeNotExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::StartTimeRaw) {
                return TString("4000000000");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};
    TDuration duration = TDuration::Seconds(60);

    auto node = CreateNode(porto, name, duration, EPortoContainerProperty::StartTimeRaw, TPortoContainerTimeExpiredNode::ETimeType::UNIX_TIMESTAMP);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestCreationTimeNotExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::CreationTimeRaw) {
                return TString("4000000000");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};
    TDuration duration = TDuration::Seconds(60);

    auto node = CreateNode(porto, name, duration, EPortoContainerProperty::CreationTimeRaw, TPortoContainerTimeExpiredNode::ETimeType::UNIX_TIMESTAMP);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestExecutionTimeNotExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::Time) {
                return TString("30");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};
    TDuration duration = TDuration::Seconds(60);

    auto node = CreateNode(porto, name, duration, EPortoContainerProperty::Time, TPortoContainerTimeExpiredNode::ETimeType::ABSOLUTE_VALUE);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestStartTimeNowNotExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TMyPortoClient(const TString& time) : Time_(time) {}

        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::StartTimeRaw) {
                return Time_;
            }
            return TPortoError();
        }

        TString Time_;
        TPortoContainerName Name_ = TString("");
    };

    TInstant now = Now();
    TPortoClientPtr myPorto = new TMyPortoClient(ToString(now.Seconds()));
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};
    TDuration duration = TDuration::Seconds(60);

    auto node = CreateNode(porto, name, duration, EPortoContainerProperty::StartTimeRaw, TPortoContainerTimeExpiredNode::ETimeType::UNIX_TIMESTAMP);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestCreationTimeNowNotExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TMyPortoClient(const TString& time) : Time_(time) {}

        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::CreationTimeRaw) {
                return Time_;
            }
            return TPortoError();
        }

        TString Time_;
        TPortoContainerName Name_ = TString("");
    };

    TInstant now = Now();
    TPortoClientPtr myPorto = new TMyPortoClient(ToString(now.Seconds()));
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};
    TDuration duration = TDuration::Seconds(60);

    auto node = CreateNode(porto, name, duration, EPortoContainerProperty::CreationTimeRaw, TPortoContainerTimeExpiredNode::ETimeType::UNIX_TIMESTAMP);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestStartTimeBadSyntax) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::StartTimeRaw) {
                return TString("bad syntax");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};
    TDuration duration = TDuration::Seconds(60);

    auto node = CreateNode(porto, name, duration, EPortoContainerProperty::StartTimeRaw, TPortoContainerTimeExpiredNode::ETimeType::UNIX_TIMESTAMP);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "bad syntax");
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestCreationTimeBadSyntax) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::CreationTimeRaw) {
                return TString("bad syntax");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};
    TDuration duration = TDuration::Seconds(60);

    auto node = CreateNode(porto, name, duration, EPortoContainerProperty::CreationTimeRaw, TPortoContainerTimeExpiredNode::ETimeType::UNIX_TIMESTAMP);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "bad syntax");
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestExecutionTimeBadSyntax) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::Time) {
                return TString("bad syntax");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};
    TDuration duration = TDuration::Seconds(60);

    auto node = CreateNode(porto, name, duration, EPortoContainerProperty::Time, TPortoContainerTimeExpiredNode::ETimeType::ABSOLUTE_VALUE);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "bad syntax");
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoContainerTimeExpiredNode
