#include "porto_create_volume_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/box_status_repository.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/layer_status_repository.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/static_resource_status_repository.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/volume_status_repository.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoCreateVolume {

class ITestPortoCreateVolumeNodeCanon {
public:
    ITestPortoCreateVolumeNodeCanon(TPortoClientPtr porto)
        : Layers_(TVector<TString>({"some_layer", "other_layer"}))
        , StaticResourceOriginalPaths_(TVector<TString>({"resource1", "resource2"}))
        , StaticResourcePaths_(TVector<TString>({"resource1_mountpoint", "resoure2_mountpoint"}))
        , BoxStatusRepository_(new TBoxStatusRepository())
        , LayerStatusRepository_(new TLayerStatusRepository())
        , StaticResourceStatusRepository_(new TStaticResourceStatusRepository())
        , VolumeStatusRepository_(new TVolumeStatusRepository())
        , WorkloadStatusRepository_(new TWorkloadStatusRepository())
        , MyPorto_(porto)
        , Porto_(new TAsyncPortoClient(porto, new TFakeThreadPool()))
    {}

    virtual ~ITestPortoCreateVolumeNodeCanon() {
    }

    void DoTest() {
        Test();
    }

    TPortoCreateVolumeNodePtr CreateNode(const TString& objectId, TStatusRepositoryCommonPtr statusRepository) const {
        return new TPortoCreateVolumeNode(
            {1, "title"}
            , Porto_
            , statusRepository
            , objectId
            , PATH
            , STORAGE
            , PLACE
            , Layers_
            , StaticResourceOriginalPaths_
            , StaticResourcePaths_
            , QUOTA_BYTES
            , PRIVATE
            , BACKEND
            , READ_ONLY
        );
    }

protected:
    virtual void Test() = 0;

protected:
    TVector<TString> Layers_;
    TVector<TString> StaticResourceOriginalPaths_;
    TVector<TString> StaticResourcePaths_;

    TBoxStatusRepositoryPtr BoxStatusRepository_;
    TLayerStatusRepositoryPtr LayerStatusRepository_;
    TStaticResourceStatusRepositoryPtr StaticResourceStatusRepository_;
    TVolumeStatusRepositoryPtr VolumeStatusRepository_;
    TWorkloadStatusRepositoryPtr WorkloadStatusRepository_;
    TPortoClientPtr MyPorto_;
    TAsyncPortoClientPtr Porto_;

public:
    static const TString PATH;
    static const TPortoContainerName CONTAINER_NAME;
    static const TString STORAGE;
    static const TString PLACE;
    static const size_t QUOTA_BYTES;
    static const TString PRIVATE;
    static const EPortoVolumeBackend BACKEND;
    static const bool READ_ONLY;
};

const TString ITestPortoCreateVolumeNodeCanon::PATH = "/some_path";
const TPortoContainerName ITestPortoCreateVolumeNodeCanon::CONTAINER_NAME = TPortoContainerName::NoEscape(".");
const TString ITestPortoCreateVolumeNodeCanon::STORAGE = "my_storage";
const TString ITestPortoCreateVolumeNodeCanon::PLACE = "my_place";
const size_t ITestPortoCreateVolumeNodeCanon::QUOTA_BYTES = (1 << 30);
const TString ITestPortoCreateVolumeNodeCanon::PRIVATE = "my_private";
const EPortoVolumeBackend ITestPortoCreateVolumeNodeCanon::BACKEND = EPortoVolumeBackend::Bind;
const bool ITestPortoCreateVolumeNodeCanon::READ_ONLY = true;

Y_UNIT_TEST_SUITE(PortoCreateVolumeNodeSuite) {

static TLogger logger({});

Y_UNIT_TEST(TestCreateFailureWhenStaticResourceDataIsInvalid) {
    class TTest : public ITestPortoCreateVolumeNodeCanon {
    public:
        TTest(TPortoClientPtr porto) : ITestPortoCreateVolumeNodeCanon(porto)
        {}

    protected:
        void Test() override {
            const TString volumeId = "my_volume";
            const TString workloadId = "my_workload";

            VolumeStatusRepository_->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(volumeId));
            WorkloadStatusRepository_->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

            UNIT_ASSERT_EXCEPTION_CONTAINS(CreateNode(workloadId, WorkloadStatusRepository_), yexception, "You can't create volume in workload tree");

            StaticResourceOriginalPaths_.pop_back();
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                CreateNode(
                    volumeId
                    , VolumeStatusRepository_
                )
                , yexception
                , "TPortoCreateVolumeNode static resource paths size should be equal to static resource original paths size"
            );
        }
    };

    TTest test(new TMockPortoClient());
    test.DoTest();
}

Y_UNIT_TEST(TestCreateSuccess) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> CreateVolume(
            const TString& path
            , const TString& storage
            , const TString& place
            , const TVector<TString>& layers
            , unsigned long long quotaBytes
            , const TString& privateValue
            , const EPortoVolumeBackend backend
            , const TPortoContainerName& containerName
            , const TVector<TPortoVolumeShare>& staticResources
            , bool readOnly
        ) override {
            ++Calls_;
            Path_ = path;
            Storage_ = storage;
            Place_ = place;
            Layers_ = layers;
            QuotaBytes_ = quotaBytes;
            PrivateValue_ = privateValue;
            Backend_ = backend;
            ContainerName_ = containerName;
            StaticResources_ = staticResources;
            ReadOnly_ = readOnly;

            return TString("/generated_path");
        }

        size_t Calls_ = 0;
        TString Path_ = "";
        TString Storage_ = "";
        TString Place_ = "";
        TVector<TString> Layers_;
        unsigned long long QuotaBytes_;
        TString PrivateValue_;
        EPortoVolumeBackend Backend_;
        TPortoContainerName ContainerName_ = ITestPortoCreateVolumeNodeCanon::CONTAINER_NAME;
        TVector<TPortoVolumeShare> StaticResources_;
        bool ReadOnly_;
    };

    class TTest : public ITestPortoCreateVolumeNodeCanon {
    public:
        TTest(TPortoClientPtr porto) : ITestPortoCreateVolumeNodeCanon(porto)
        {}

    protected:
        void Test() override {
            const TString volumeId = "my_volume";
            VolumeStatusRepository_->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(volumeId));
            auto node = CreateNode(volumeId, VolumeStatusRepository_);

            auto result = node->Tick(MockTickContext(logger));
            UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
            UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)MyPorto_.Get())->Calls_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::PATH, ((TMyPortoClient*)MyPorto_.Get())->Path_);
            UNIT_ASSERT_EQUAL(Layers_.at(0), ((TMyPortoClient*)MyPorto_.Get())->Layers_.at(0));
            UNIT_ASSERT_EQUAL(Layers_.at(1), ((TMyPortoClient*)MyPorto_.Get())->Layers_.at(1));
            UNIT_ASSERT_EQUAL(StaticResourcePaths_.at(0), ((TMyPortoClient*)MyPorto_.Get())->StaticResources_.at(0).Path);
            UNIT_ASSERT_EQUAL(StaticResourcePaths_.at(1), ((TMyPortoClient*)MyPorto_.Get())->StaticResources_.at(1).Path);
            UNIT_ASSERT_EQUAL(StaticResourceOriginalPaths_.at(0), ((TMyPortoClient*)MyPorto_.Get())->StaticResources_.at(0).OriginPath);
            UNIT_ASSERT_EQUAL(StaticResourceOriginalPaths_.at(1), ((TMyPortoClient*)MyPorto_.Get())->StaticResources_.at(1).OriginPath);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::QUOTA_BYTES, ((TMyPortoClient*)MyPorto_.Get())->QuotaBytes_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::PRIVATE, ((TMyPortoClient*)MyPorto_.Get())->PrivateValue_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::BACKEND, ((TMyPortoClient*)MyPorto_.Get())->Backend_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::STORAGE, ((TMyPortoClient*)MyPorto_.Get())->Storage_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::PLACE, ((TMyPortoClient*)MyPorto_.Get())->Place_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::READ_ONLY, ((TMyPortoClient*)MyPorto_.Get())->ReadOnly_);
            UNIT_ASSERT_EQUAL(((TMyPortoClient*)MyPorto_.Get())->ContainerName_, ITestPortoCreateVolumeNodeCanon::CONTAINER_NAME);
        }
    };

    TTest test(new TMyPortoClient());
    test.DoTest();
}

Y_UNIT_TEST(TestCreateVolumeWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<TString, TPortoError> CreateVolume(
            const TString& path
            , const TString& storage
            , const TString& place
            , const TVector<TString>& layers
            , unsigned long long quotaBytes
            , const TString& privateValue
            , const EPortoVolumeBackend backend
            , const TPortoContainerName& containerName
            , const TVector<TPortoVolumeShare>& staticResources
            , bool readOnly
        ) override {
            ++Calls_;
            Path_ = path;
            Storage_ = storage;
            Place_ = place;
            Layers_ = layers;
            QuotaBytes_ = quotaBytes;
            PrivateValue_ = privateValue;
            Backend_ = backend;
            ContainerName_ = containerName;
            StaticResources_ = staticResources;
            ReadOnly_ = readOnly;

            return TPortoError{EPortoError::Unknown, "CreateVolume", "NO"};
        }

        size_t Calls_ = 0;
        TString Path_ = "";
        TString Storage_ = "";
        TString Place_ = "";
        TVector<TString> Layers_;
        unsigned long long QuotaBytes_;
        TString PrivateValue_;
        EPortoVolumeBackend Backend_;
        TPortoContainerName ContainerName_ = {""};
        TVector<TPortoVolumeShare> StaticResources_;
        bool ReadOnly_;
    };

    class TTest : public ITestPortoCreateVolumeNodeCanon {
    public:
        TTest(TPortoClientPtr porto) : ITestPortoCreateVolumeNodeCanon(porto)
        {}

    protected:
        void CheckClient() {
            UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)MyPorto_.Get())->Calls_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::PATH, ((TMyPortoFailingClient*)MyPorto_.Get())->Path_);
            UNIT_ASSERT_EQUAL(Layers_.at(0), ((TMyPortoFailingClient*)MyPorto_.Get())->Layers_.at(0));
            UNIT_ASSERT_EQUAL(Layers_.at(1), ((TMyPortoFailingClient*)MyPorto_.Get())->Layers_.at(1));
            UNIT_ASSERT_EQUAL(StaticResourcePaths_.at(0), ((TMyPortoFailingClient*)MyPorto_.Get())->StaticResources_.at(0).Path);
            UNIT_ASSERT_EQUAL(StaticResourcePaths_.at(1), ((TMyPortoFailingClient*)MyPorto_.Get())->StaticResources_.at(1).Path);
            UNIT_ASSERT_EQUAL(StaticResourceOriginalPaths_.at(0), ((TMyPortoFailingClient*)MyPorto_.Get())->StaticResources_.at(0).OriginPath);
            UNIT_ASSERT_EQUAL(StaticResourceOriginalPaths_.at(1), ((TMyPortoFailingClient*)MyPorto_.Get())->StaticResources_.at(1).OriginPath);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::QUOTA_BYTES, ((TMyPortoFailingClient*)MyPorto_.Get())->QuotaBytes_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::PRIVATE, ((TMyPortoFailingClient*)MyPorto_.Get())->PrivateValue_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::BACKEND, ((TMyPortoFailingClient*)MyPorto_.Get())->Backend_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::STORAGE, ((TMyPortoFailingClient*)MyPorto_.Get())->Storage_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::PLACE, ((TMyPortoFailingClient*)MyPorto_.Get())->Place_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::READ_ONLY, ((TMyPortoFailingClient*)MyPorto_.Get())->ReadOnly_);
            UNIT_ASSERT_EQUAL(((TMyPortoFailingClient*)MyPorto_.Get())->ContainerName_, ITestPortoCreateVolumeNodeCanon::CONTAINER_NAME);
        }

        void Test() override {
            {
                ((TMyPortoFailingClient*)MyPorto_.Get())->Calls_ = 0;
                const TString staticResourceId = "my_static_resource";
                const TString staticResourceDownloadHash = "my_static_resource_download_hash";
                StaticResourceStatusRepository_->AddObject(NObjectMetaTestLib::CreateStaticResourceMetaSimple(staticResourceId, staticResourceDownloadHash));

                auto node = CreateNode(staticResourceDownloadHash, StaticResourceStatusRepository_);
                auto result = node->Tick(MockTickContext(logger));

                UNIT_ASSERT_C(!result, result.Success().Message);
                UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "CreateVolume");
                UNIT_ASSERT_EQUAL(StaticResourceStatusRepository_->GetObjectStatus(staticResourceId).failed().message(), result.Error().Message);
                UNIT_ASSERT_EQUAL(StaticResourceStatusRepository_->GetObjectStatus(staticResourceId).fail_counter(), 1);

                CheckClient();
            }

            {
                ((TMyPortoFailingClient*)MyPorto_.Get())->Calls_ = 0;
                const TString layerId = "my_layer";
                const TString layerDownloadHash = "my_layer_download_hash";
                LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerDownloadHash));

                auto node = CreateNode(layerDownloadHash, LayerStatusRepository_);
                auto result = node->Tick(MockTickContext(logger));

                UNIT_ASSERT_C(!result, result.Success().Message);
                UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "CreateVolume");
                UNIT_ASSERT_EQUAL(LayerStatusRepository_->GetObjectStatus(layerId).failed().message(), result.Error().Message);
                UNIT_ASSERT_EQUAL(LayerStatusRepository_->GetObjectStatus(layerId).fail_counter(), 1);

                CheckClient();
            }

            {
                ((TMyPortoFailingClient*)MyPorto_.Get())->Calls_ = 0;
                const TString volumeId = "my_volume";
                VolumeStatusRepository_->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(volumeId));

                auto node = CreateNode(volumeId, VolumeStatusRepository_);
                auto result = node->Tick(MockTickContext(logger));

                UNIT_ASSERT_C(!result, result.Success().Message);
                UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "CreateVolume");
                UNIT_ASSERT_EQUAL(VolumeStatusRepository_->GetObjectStatus(volumeId).failed().message(), result.Error().Message);
                UNIT_ASSERT_EQUAL(VolumeStatusRepository_->GetObjectStatus(volumeId).fail_counter(), 1);

                CheckClient();
            }

            {
                ((TMyPortoFailingClient*)MyPorto_.Get())->Calls_ = 0;
                const TString boxId = "my_box";
                BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));

                auto node = CreateNode(boxId, BoxStatusRepository_);
                auto result = node->Tick(MockTickContext(logger));

                UNIT_ASSERT_C(!result, result.Success().Message);
                UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "CreateVolume");
                UNIT_ASSERT_EQUAL(BoxStatusRepository_->GetObjectStatus(boxId).failed().message(), result.Error().Message);
                UNIT_ASSERT_EQUAL(BoxStatusRepository_->GetObjectStatus(boxId).fail_counter(), 1);

                CheckClient();
            }
        }
    };

    TTest test(new TMyPortoFailingClient());
    test.DoTest();
}


Y_UNIT_TEST(TestCreateVolumeWithFailingCreate) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<TString, TPortoError> CreateVolume(
            const TString& path
            , const TString& storage
            , const TString& place
            , const TVector<TString>& layers
            , unsigned long long quotaBytes
            , const TString& privateValue
            , const EPortoVolumeBackend backend
            , const TPortoContainerName& containerName
            , const TVector<TPortoVolumeShare>& staticResources
            , bool readOnly
        ) override {
            ++Calls_;
            Path_ = path;
            Storage_ = storage;
            Place_ = place;
            Layers_ = layers;
            QuotaBytes_ = quotaBytes;
            PrivateValue_ = privateValue;
            Backend_ = backend;
            ContainerName_ = containerName;
            StaticResources_ = staticResources;
            ReadOnly_ = readOnly;

            return TPortoError{EPortoError::Busy, "CreateVolume", "NO"};
        }

        size_t Calls_ = 0;
        TString Path_ = "";
        TString Storage_ = "";
        TString Place_ = "";
        TVector<TString> Layers_;
        unsigned long long QuotaBytes_;
        TString PrivateValue_;
        EPortoVolumeBackend Backend_;
        TPortoContainerName ContainerName_ = {""};
        TVector<TPortoVolumeShare> StaticResources_;
        bool ReadOnly_;
    };

    class TTest : public ITestPortoCreateVolumeNodeCanon {
    public:
        TTest(TPortoClientPtr porto) : ITestPortoCreateVolumeNodeCanon(porto)
        {}

    protected:
        void Test() override {
            ((TMyPortoFailingClient*)MyPorto_.Get())->Calls_ = 0;
            const TString volumeId = "my_volume";
            VolumeStatusRepository_->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(volumeId));

            auto node = CreateNode(volumeId, VolumeStatusRepository_);
            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_C(result, result.Error().Message);
            UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
            UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "CreateVolume");
            UNIT_ASSERT_EQUAL(VolumeStatusRepository_->GetObjectStatus(volumeId).failed().message(), result.Success().Message);
            UNIT_ASSERT_EQUAL(VolumeStatusRepository_->GetObjectStatus(volumeId).fail_counter(), 1);

            UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)MyPorto_.Get())->Calls_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::PATH, ((TMyPortoFailingClient*)MyPorto_.Get())->Path_);
            UNIT_ASSERT_EQUAL(Layers_.at(0), ((TMyPortoFailingClient*)MyPorto_.Get())->Layers_.at(0));
            UNIT_ASSERT_EQUAL(Layers_.at(1), ((TMyPortoFailingClient*)MyPorto_.Get())->Layers_.at(1));
            UNIT_ASSERT_EQUAL(StaticResourcePaths_.at(0), ((TMyPortoFailingClient*)MyPorto_.Get())->StaticResources_.at(0).Path);
            UNIT_ASSERT_EQUAL(StaticResourcePaths_.at(1), ((TMyPortoFailingClient*)MyPorto_.Get())->StaticResources_.at(1).Path);
            UNIT_ASSERT_EQUAL(StaticResourceOriginalPaths_.at(0), ((TMyPortoFailingClient*)MyPorto_.Get())->StaticResources_.at(0).OriginPath);
            UNIT_ASSERT_EQUAL(StaticResourceOriginalPaths_.at(1), ((TMyPortoFailingClient*)MyPorto_.Get())->StaticResources_.at(1).OriginPath);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::QUOTA_BYTES, ((TMyPortoFailingClient*)MyPorto_.Get())->QuotaBytes_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::PRIVATE, ((TMyPortoFailingClient*)MyPorto_.Get())->PrivateValue_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::BACKEND, ((TMyPortoFailingClient*)MyPorto_.Get())->Backend_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::STORAGE, ((TMyPortoFailingClient*)MyPorto_.Get())->Storage_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::PLACE, ((TMyPortoFailingClient*)MyPorto_.Get())->Place_);
            UNIT_ASSERT_EQUAL(ITestPortoCreateVolumeNodeCanon::READ_ONLY, ((TMyPortoFailingClient*)MyPorto_.Get())->ReadOnly_);
            UNIT_ASSERT_EQUAL(((TMyPortoFailingClient*)MyPorto_.Get())->ContainerName_, ITestPortoCreateVolumeNodeCanon::CONTAINER_NAME);
        }
    };

    TTest test(new TMyPortoFailingClient());
    test.DoTest();
}

}

} //namespace NInfra::NPodAgent::NTestPortoCreateVolume
