#include "porto_destroy_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoDestroyNode {

Y_UNIT_TEST_SUITE(PortoDestroyNodeSuite) {

static TLogger logger({});

TPortoDestroyNodePtr PortoCreator(TPortoClientPtr porto, const TPortoContainerName& name) {
    return new TPortoDestroyNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , new TAsyncPortoClient(porto, new TFakeThreadPool())
        , name
    );
}

Y_UNIT_TEST(TestDestroyContainer) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> Destroy(const TPortoContainerName& name) override {
            ++Calls_;
            LastName_ = name;

            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
    };

    TPortoClientPtr porto = new TMyPortoClient();
    TPortoContainerName name = {"some_name"};

    auto creator = PortoCreator(porto, name);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)porto.Get())->LastName_);
}

Y_UNIT_TEST(TestDestroyContainerWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> Destroy(const TPortoContainerName& name) override {
            ++Calls_;
            LastName_ = name;

            return TPortoError{EPortoError::Unknown, "Destroy", "NO"};
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
    };

    TPortoClientPtr porto = new TMyPortoFailingClient();
    TPortoContainerName name = {"some_name"};

    auto creator = PortoCreator(porto, name);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "Destroy");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoFailingClient*)porto.Get())->LastName_);
}

Y_UNIT_TEST(TestDestroyContainerDoesNotExist) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> Destroy(const TPortoContainerName& name) override {
            ++Calls_;
            LastName_ = name;

            return TPortoError{EPortoError::ContainerDoesNotExist, "Destroy", "NO"};
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
    };

    TPortoClientPtr porto = new TMyPortoClient();
    TPortoContainerName name = {"some_name"};

    auto creator = PortoCreator(porto, name);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "Already destroyed");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)porto.Get())->LastName_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoDestroyNode
