#include "porto_get_and_check_properties_node.h"

#include <contrib/libs/re2/re2/re2.h>

#include <infra/pod_agent/libs/common/common.h>
#include <infra/pod_agent/libs/util/string_utils.h>

#include <library/cpp/digest/md5/md5.h>

#include <util/generic/ymath.h>
#include <util/string/type.h>

namespace NInfra::NPodAgent {

ENodeType TPortoGetAndCheckPropertiesNode::GetType() const {
    return TPortoGetAndCheckPropertiesNode::NODE_TYPE;
}

TExpected<TFuture<TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError>>, TTickResult> TPortoGetAndCheckPropertiesNode::PortoCall(TTickContextPtr /*context*/) {
    TVector<EPortoContainerProperty> properties;
    properties.reserve(Properties_.size() + 1);
    for (auto& it : Properties_) {
        properties.push_back(it.first);
    }
    properties.push_back(EPortoContainerProperty::Private);
    return Porto_->Get({ContainerName_}, properties);
}

TTickResult TPortoGetAndCheckPropertiesNode::ProcessPortoResultSuccess(TTickContextPtr /*context*/, TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>& result) {
    if (!result.contains(ContainerName_)) {
        return TNodeError({TStringBuilder() << "porto get call doesn't contain " << Quote(TString(ContainerName_)) << " container"});
    }
    TMap<EPortoContainerProperty, TPortoGetResponse>& actualProps = result[ContainerName_];
    for (auto& it : Properties_) {
        if (actualProps[it.first].error() == EPortoError::ContainerDoesNotExist) {
            return TNodeSuccess(ENodeStatus::FAILURE, TStringBuilder() << "container " << Quote(TString(ContainerName_)) << " does not exist");
        }
        if (actualProps[it.first].error()) {
            return TNodeError({TStringBuilder() << ToString(actualProps[it.first].error()) << ':' << actualProps[it.first].errormsg()});
        }

        const TString& actualValue = actualProps[it.first].value();
        const TString& inputValue = it.second;

        OUTCOME_TRYV(ComparePropertyValues(it.first, inputValue, actualValue));
    }

    if (actualProps[EPortoContainerProperty::Private].error()) {
        return TNodeError{
            TStringBuilder()
                << ToString(actualProps[EPortoContainerProperty::Private].error()) << ':'
                << actualProps[EPortoContainerProperty::Private].errormsg()
        };
    }

    TString treeHash = UnpackContainerPrivate(actualProps[EPortoContainerProperty::Private].value()).TreeHash_;
    if (treeHash != TreeHash_) {
        return TNodeSuccess(
            ENodeStatus::FAILURE
            , TStringBuilder()
                << "'tree_hash' property expected " << Quote(TreeHash_)
                << " got " << Quote(treeHash)
        );
    }

    return TNodeSuccess(ENodeStatus::SUCCESS);
}

TTickResult TPortoGetAndCheckPropertiesNode::ProcessPortoResultError(TTickContextPtr context, TPortoError& result) {
    // Process error
    auto pResult = TPortoBasicContainerNode<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>>::ProcessPortoResultError(context, result);

    // Special cases
    if (result.Code == EPortoError::Busy
        || result.Code == EPortoError::ContainerDoesNotExist
    ) {
        return TNodeSuccess(ENodeStatus::FAILURE, ToString(result));
    } else {
        return pResult;
    }
}

TExpected<void, TTickResult> TPortoGetAndCheckPropertiesNode::ComparePropertyValues(
    EPortoContainerProperty property
    , const TString& inputValue
    , const TString& actualValue
) const {
    bool isValueSecret = SecretProperties_.contains(property);
    switch(property) {
        case EPortoContainerProperty::Ulimit:
            OUTCOME_TRYV(CompareUlimit(inputValue, actualValue));
            break;
        case EPortoContainerProperty::CpuLimit:
        case EPortoContainerProperty::CpuGuarantee:
            OUTCOME_TRYV(CompareDouble(property, inputValue, actualValue, 0.0, "c"));
            break;
        case EPortoContainerProperty::CpuWeight:
            // TODO(DEPLOY-2046, chegoryu) set default inputValue to 1.0 (not actualValue)
            {
                double actualValueDouble = OUTCOME_TRYX(GetDoubleValue(property, actualValue, 1.0, ""));
                OUTCOME_TRYV(CompareDouble(property, inputValue, actualValue, actualValueDouble, ""));
            }
            break;
        case EPortoContainerProperty::CpuPolicy:
            // If cpu_policy not set - every value is correct
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, actualValue, isValueSecret));
            break;
        case EPortoContainerProperty::MemoryLimit:
        case EPortoContainerProperty::MemoryGuarantee:
        case EPortoContainerProperty::AnonLimit:
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, "0", isValueSecret));
            break;
        case EPortoContainerProperty::RechPgfault:
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, "false", isValueSecret));
            break;
        case EPortoContainerProperty::Cwd:
            // if cwd not set - every cwd is correct
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, actualValue, isValueSecret));
            break;
        case EPortoContainerProperty::Root:
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, "/", isValueSecret));
            break;
        case EPortoContainerProperty::Place:
            // if place not set - every place is correct
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, actualValue, isValueSecret));
            break;
        case EPortoContainerProperty::User:
        case EPortoContainerProperty::Group:
            // if user or group not set - every user or group is correct
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, actualValue, isValueSecret));
            break;
        case EPortoContainerProperty::StdOutPath:
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, "stdout", isValueSecret));
            break;
        case EPortoContainerProperty::StdErrPath:
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, "stderr", isValueSecret));
            break;
        case EPortoContainerProperty::StdOutAndStdErrLimit:
            // if limit not set - every limit is correct, because it is random big number
            // On porto 4.18.20 limit is 8388608
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, actualValue, isValueSecret));
            break;
        case EPortoContainerProperty::ResolvConf:
            // If resolv_conf not set - every value is correct
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, actualValue, isValueSecret));
            break;
        case EPortoContainerProperty::Hostname:
            // If hostname not set - every value is correct
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, actualValue, isValueSecret));
            break;
        case EPortoContainerProperty::ThreadLimit:
            // If thread_limit not set - every value is correct
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, actualValue, isValueSecret));
            break;
        case EPortoContainerProperty::AgingTime:
            // If aging_time not set - every value is correct
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, actualValue, isValueSecret));
            break;
        case EPortoContainerProperty::IoLimit:
        case EPortoContainerProperty::IoOpsLimit:
            OUTCOME_TRYV(CompareSemicolonSeparatedSequence(property, inputValue, actualValue, {}, true, isValueSecret));
            break;
        case EPortoContainerProperty::IoWeight:
            OUTCOME_TRYV(CompareDouble(property, inputValue, actualValue, 1.0, ""));
            break;
        case EPortoContainerProperty::IoPolicy:
            // If io_policy not set - every value is correct
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, actualValue, isValueSecret));
            break;
        case EPortoContainerProperty::EnablePorto:
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, "true", isValueSecret));
            break;
        case EPortoContainerProperty::EnvSecret:
            OUTCOME_TRYV(CompareSecrets(inputValue, actualValue, actualValue));
            break;
        case EPortoContainerProperty::Cgroupfs:
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, "none", isValueSecret));
            break;
        default:
            OUTCOME_TRYV(CompareString(property, inputValue, actualValue, "", isValueSecret));
            break;
    }

    return TExpected<void, TTickResult>::DefaultSuccess();
}

TTickResult TPortoGetAndCheckPropertiesNode::CreateFailureResult(
    EPortoContainerProperty property
    , const TString& inputValue
    , const TString& actualValue
    , const TString& message
    , const bool isValueSecret
) const {
    const TString& patchedInputValue = isValueSecret ? "<Value is hidden>" : inputValue;
    const TString& patchedActualValue = isValueSecret ? "<Value is hidden>" : actualValue;

    TStringBuilder failureMessage = TStringBuilder()
        << Quote(ToString(property)) << " property expected " << Quote(patchedInputValue)
        << " got " << Quote(patchedActualValue);
    if (!message.empty()) {
        failureMessage << ". Compare message: " << Quote(message);
    }
    return TNodeSuccess(ENodeStatus::FAILURE, failureMessage);
}

TExpected<void, TTickResult> TPortoGetAndCheckPropertiesNode::CompareUlimit(
    const TString& inputValue
    , const TString& actualValue
) const {
    TString inputValueCopy = inputValue;

    // remove empty Ulimit properties
    re2::RE2::Replace(&inputValueCopy, "[a-z]+: ; ", "");

    return CompareString(EPortoContainerProperty::Ulimit, inputValueCopy, actualValue, "", false);
}

TExpected<void, TTickResult> TPortoGetAndCheckPropertiesNode::CompareString(
    EPortoContainerProperty property
    , const TString& inputValue
    , const TString& actualValue
    , const TString& defaultOnInputValueEmpty
    , const bool isValueSecret
) const {
    if (!inputValue.empty()) {
        if (inputValue != actualValue) {
            return CreateFailureResult(property, inputValue, actualValue, "", isValueSecret);
        }
    } else {
        if (defaultOnInputValueEmpty != actualValue) {
            return CreateFailureResult(
                property
                , inputValue
                , actualValue
                , TStringBuilder() << "Treat empty value as " << Quote(defaultOnInputValueEmpty)
                , isValueSecret
            );
        }
    }

    return TExpected<void, TTickResult>::DefaultSuccess();
}

TExpected<void, TTickResult> TPortoGetAndCheckPropertiesNode::CompareDouble(
    EPortoContainerProperty property
    , const TString& inputValue
    , const TString& actualValue
    , const double defaultOnInputValueEmpty
    , const TString& suffix
) const {
    double inputValueDouble = OUTCOME_TRYX(GetDoubleValue(property, inputValue, defaultOnInputValueEmpty, suffix));
    double actualValueDouble = OUTCOME_TRYX(GetDoubleValue(property, actualValue, defaultOnInputValueEmpty, suffix));

    if (Abs(inputValueDouble - actualValueDouble) >= TPortoGetAndCheckPropertiesNode::DOUBLE_COMPARISON_PRECISION) {
       return CreateFailureResult(
            property
            , inputValue
            , actualValue
            , TStringBuilder() << "Compare as double values, error '" << Abs(inputValueDouble - actualValueDouble)
                               << "' allowable error '" << TPortoGetAndCheckPropertiesNode::DOUBLE_COMPARISON_PRECISION << "'"
            , false
        );
    }

    return TExpected<void, TTickResult>::DefaultSuccess();
}

TExpected<double, TTickResult> TPortoGetAndCheckPropertiesNode::GetDoubleValue(
    EPortoContainerProperty property
    , const TString& stringValue
    , const double defaultOnStringValueEmpty
    , const TString& suffix
) const {
    if (stringValue.empty()) {
        return defaultOnStringValueEmpty;
    }

    if (!stringValue.EndsWith(suffix)) {
        return TTickResult(TNodeError{
            TStringBuilder()
                << "Suffix " << Quote(suffix)
                << " expected for " << Quote(ToString(property))
                << " but actual value is " << Quote(stringValue)
        });
    }

    try {
        double doubleValue = FromString<double>(stringValue.substr(0, stringValue.size() - suffix.size()));
        return doubleValue;
    } catch (const yexception& e) {
        return TTickResult(TNodeError{
            TStringBuilder()
                << "Error while converting " << Quote(stringValue)
                << " to double " << Quote(e.what())
                << " in property " << Quote(ToString(property))
        });
    }
}

TExpected<void, TTickResult> TPortoGetAndCheckPropertiesNode::CompareSemicolonSeparatedSequence(
    EPortoContainerProperty property
    , const TString& inputValue
    , const TString& actualValue
    , const TVector<TString>& defaultOnInputValueEmpty
    , const bool stripValues
    , const bool isValueSecret
) const {
    TVector<TString> inputValueVector = OUTCOME_TRYX(GetSemicolonSeparatedSequenceVector(inputValue, defaultOnInputValueEmpty, stripValues));
    TVector<TString> actualValueVector = OUTCOME_TRYX(GetSemicolonSeparatedSequenceVector(actualValue, defaultOnInputValueEmpty, stripValues));

    if (inputValueVector != actualValueVector) {
        return CreateFailureResult(
            property
            , inputValue
            , actualValue
            , "Compare as semicolon separated sequence, the order of elements is not taken into account"
            , isValueSecret
        );
    }

    return TExpected<void, TTickResult>::DefaultSuccess();
}

TExpected<TVector<TString>, TTickResult> TPortoGetAndCheckPropertiesNode::GetSemicolonSeparatedSequenceVector(
    const TString& stringValue
    , const TVector<TString>& defaultOnStringValueEmpty
    , const bool stripValues
) const {
    if (stringValue.empty()) {
        return defaultOnStringValueEmpty;
    }

    TVector<TString> result;

    char lastSymb = ';';
    TString currentToken = "";
    for (auto symb : stringValue) {
        if (symb == ';' && lastSymb != '\\') {
            if (!currentToken.empty()) {
                result.push_back(currentToken); // skip empty tokens
            }
            currentToken = "";
        } else {
            if (symb == ';' && lastSymb == '\\') {
                currentToken.pop_back(); // Remove escape character
            }

            currentToken.push_back(symb);
        }

        lastSymb = symb;
    }
    if (!currentToken.empty()) {
        result.push_back(currentToken);
    }

    if (stripValues) {
        TVector<TString> stripedResult;
        for (const TString& token : result) {
            TString stripedToken = token;
            while (!stripedToken.empty() && IsSpace(stripedToken.substr(stripedToken.size() - 1, 1))) {
                stripedToken.pop_back();
            }

            size_t ptr = 0;
            while (ptr < stripedToken.size() && IsSpace(stripedToken.substr(ptr, 1))) {
                ++ptr;
            }
            if (ptr < stripedToken.size()) {
                stripedResult.push_back(stripedToken.substr(ptr)); // skip empty tokens
            }
        }

        result = stripedResult;
    }

    // Normalize result by sorting because order of the elements is not important
    Sort(result.begin(), result.end());

    return result;
}

TExpected<TMap<TString, TString>, TTickResult> TPortoGetAndCheckPropertiesNode::GetEnvMap(const TString& envStr) {
    TMap<TString, TString> result;
    TVector<TString> envs = SplitEscaped(envStr, ';');
    for (const auto& env : envs) {
        size_t sep = env.find('=');
        if (sep == TString::npos) {
            return TTickResult(TNodeError{
                TStringBuilder()
                    << "Error while converting value to map of envs in property "
                    << Quote(ToString(EPortoContainerProperty::EnvSecret))
            });
        }
        result[Trim(env.substr(0, sep))] = Trim(env.substr(sep + 1));
    }
    return result;
}

TExpected<void, TTickResult> TPortoGetAndCheckPropertiesNode::CompareSecrets(
    const TString& inputValue
    , const TString& actualValue
    , const TString& defaultOnInputValueEmpty
) const {
    if (inputValue.empty()) {
        if (defaultOnInputValueEmpty != actualValue) {
            return CreateFailureResult(
                EPortoContainerProperty::EnvSecret
                , inputValue
                , actualValue
                , TStringBuilder() << "Treat empty value as '" << defaultOnInputValueEmpty << "'"
                , true
            );
        }
    }

    auto inputMap = OUTCOME_TRYX(GetEnvMap(inputValue));
    auto actualMap = OUTCOME_TRYX(GetEnvMap(actualValue));
    if (inputMap.size() != actualMap.size()) {
        return CreateFailureResult(EPortoContainerProperty::EnvSecret, inputValue, actualValue, "Sizes of envs list are different", true);
    }

    for (const auto& [key, value]: actualMap) {
        if (!inputMap.contains(key)) {
            return CreateFailureResult(
                EPortoContainerProperty::EnvSecret,
                inputValue,
                actualValue,
                TStringBuilder() << "Secret " << Quote(key) << " was removed",
                true
            );
        }

        auto extractResult = ExtractSaltAndMd5FromSecret(value);
        if (!extractResult) {
            return TTickResult(TNodeError{
                TStringBuilder()
                    << "Error while parsing secret value with unexpected format in property "
                    << Quote(ToString(EPortoContainerProperty::EnvSecret))
                    << ": " << Quote(value)
            });
        }

        auto md5ToCheck = MD5::Calc(extractResult.Get()->first + inputMap[key]);
        if (extractResult.Get()->second != md5ToCheck) {
            return CreateFailureResult(
                EPortoContainerProperty::EnvSecret,
                inputValue,
                actualValue,
                TStringBuilder() << "Value of secret " << Quote(key) << " was changed",
                true
            );
        }
    }
    return TExpected<void, TTickResult>::DefaultSuccess();
}

} // namespace NInfra::NPodAgent
