#pragma once

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/porto_basic_container_node.h>
#include <infra/pod_agent/libs/behaviour/bt/nodes/base/private_util.h>

namespace NInfra::NPodAgent {

class TPortoGetAndCheckPropertiesNode;
using TPortoGetAndCheckPropertiesNodePtr = TSimpleSharedPtr<TPortoGetAndCheckPropertiesNode>;

class TPortoGetAndCheckPropertiesNode : public TPortoBasicContainerNode<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>> {
public:
    TPortoGetAndCheckPropertiesNode(
        const TBasicTreeNodeDescriptor& descriptor
        , TAsyncPortoClientPtr porto
        , TStatusRepositoryCommonPtr containerStatusRepository
        , const NStatusRepositoryTypes::TContainerDescription& container
        , const TPortoContainerName& containerName
        , const TString& treeHash
        , const TMap<EPortoContainerProperty, TString>& properties // TMap (not Hash) for stable iteration
        , const THashSet<EPortoContainerProperty>& secretProperties
    )
        : TPortoBasicContainerNode(descriptor, porto, containerStatusRepository, container, containerName)
        , Properties_(properties)
        , SecretProperties_(secretProperties)
        , TreeHash_(treeHash)
    {
        for (EPortoContainerProperty nonHiddenProperty : {
            EPortoContainerProperty::Ulimit
            , EPortoContainerProperty::CpuLimit
            , EPortoContainerProperty::CpuGuarantee
        }) {
            Y_ENSURE(!SecretProperties_.contains(nonHiddenProperty), "Value hiding is not implemented for '" << ToString(nonHiddenProperty) << "'");
        }

        for (auto& it: Properties_) {
            Y_ENSURE(it.first != EPortoContainerProperty::Private, "TPortoGetAndCheckPropertiesNode doesn't support EPortoContainerProperty::Private");
        }
    }

    virtual ENodeType GetType() const override final;

protected:
    virtual TExpected<TFuture<TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError>>, TTickResult> PortoCall(TTickContextPtr context) override final;
    virtual TTickResult ProcessPortoResultSuccess(TTickContextPtr context, TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>& result) override final;
    virtual TTickResult ProcessPortoResultError(TTickContextPtr context, TPortoError& result) override final;

    // Return nothing if equal, otherwise return TTickResult
    TExpected<void, TTickResult> ComparePropertyValues(
        EPortoContainerProperty property
        , const TString& inputValue
        , const TString& actualValue
    ) const;

private:
    TTickResult CreateFailureResult(
        EPortoContainerProperty property
        , const TString& inputValue
        , const TString& actualValue
        , const TString& message
        , const bool isValueSecret
    ) const;

    TExpected<void, TTickResult> CompareUlimit(
        const TString& inputValue
        , const TString& actualValue
    ) const;

    TExpected<void, TTickResult> CompareString(
        EPortoContainerProperty property
        , const TString& inputValue
        , const TString& actualValue
        , const TString& defaultOnInputValueEmpty
        , const bool isValueSecret
    ) const;

    TExpected<void, TTickResult> CompareDouble(
        EPortoContainerProperty property
        , const TString& inputValue
        , const TString& actualValue
        , const double defaultOnInputValueEmpty
        , const TString& suffix
    ) const;

    TExpected<double, TTickResult> GetDoubleValue(
        EPortoContainerProperty property
        , const TString& stringValue
        , const double defaultOnStringValueEmpty
        , const TString& suffix
    ) const;

    TExpected<void, TTickResult> CompareSemicolonSeparatedSequence(
        EPortoContainerProperty property
        , const TString& inputValue
        , const TString& actualValue
        , const TVector<TString>& defaultOnInputValueEmpty
        , const bool stripValues
        , const bool isValueSecret
    ) const;

    TExpected<TVector<TString>, TTickResult> GetSemicolonSeparatedSequenceVector(
        const TString& stringValue
        , const TVector<TString>& defaultOnStringValueEmpty
        , const bool stripValues
    ) const;

    static TExpected<TMap<TString, TString>, TTickResult> GetEnvMap(const TString& envStr);

    TExpected<void, TTickResult> CompareSecrets(
        const TString& inputValue
        , const TString& actualValue
        , const TString& defaultOnInputValueEmpty
    ) const;

public:
    static constexpr const ENodeType NODE_TYPE = ENodeType::PORTO_GET_AND_CHECK_PROPERTIES;

private:
    static constexpr const double DOUBLE_COMPARISON_PRECISION = 1e-3;

    const TMap<EPortoContainerProperty, TString> Properties_;
    const THashSet<EPortoContainerProperty> SecretProperties_;
    const TString TreeHash_;
};

} // namespace NInfra::NPodAgent
