#include "porto_get_and_check_properties_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>
#include <infra/pod_agent/libs/porto_client/test_functions.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/digest/md5/md5.h>
#include <library/cpp/testing/unittest/registar.h>

#include <random>

namespace NInfra::NPodAgent::NPortoTest::NTestPortoGetAndCheckPropertiesNode {

Y_UNIT_TEST_SUITE(PortoGetAndCheckPropertyNodeSuite) {

static TLogger logger({});

class TTestPortoGetAndCheckPropertiesNode;
using TTestPortoGetAndCheckPropertiesNodePtr = TSimpleSharedPtr<TTestPortoGetAndCheckPropertiesNode>;

class TTestPortoGetAndCheckPropertiesNode: public TPortoGetAndCheckPropertiesNode {
public:
    TTestPortoGetAndCheckPropertiesNode(
        const TBasicTreeNodeDescriptor& descriptor
        , TAsyncPortoClientPtr porto
        , TStatusRepositoryCommonPtr containerStatusRepository
        , const NStatusRepositoryTypes::TContainerDescription& container
        , const TPortoContainerName& containerName
        , const TString& treeHash
        , const TMap<EPortoContainerProperty, TString>& properties
        , const THashSet<EPortoContainerProperty>& secretProperties

    )
        : TPortoGetAndCheckPropertiesNode(
            descriptor
            , porto
            , containerStatusRepository
            , container
            , containerName
            , treeHash
            , properties
            , secretProperties
        )
    {
    }

    TExpected<void, TTickResult> ComparePropertyValues(
        EPortoContainerProperty property
        , const TString& inputValue
        , const TString& actualValue
    ) const {
        return TPortoGetAndCheckPropertiesNode::ComparePropertyValues(property, inputValue, actualValue);
    }
};

TTestPortoGetAndCheckPropertiesNodePtr CreateNode(
    TPortoClientPtr porto
    , TWorkloadStatusRepositoryPtr statusRepository
    , const TString& workloadId
    , const TPortoContainerName& containerName
    , const TString& treeHash
    , const TMap<EPortoContainerProperty, TString>& properties
    , const THashSet<EPortoContainerProperty> secretProperties
) {
    return new TTestPortoGetAndCheckPropertiesNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , new TAsyncPortoClient(porto, new TFakeThreadPool())
        , statusRepository
        , NStatusRepositoryTypes::TContainerDescription(
            workloadId
            , NStatusRepositoryTypes::EObjectType::WORKLOAD
            , NStatusRepositoryTypes::TContainerDescription::EContainerType::START
        )
        , containerName
        , treeHash
        , properties
        , secretProperties
    );
}

Y_UNIT_TEST(TestGetAndCheckProperty) {
    TMap<EPortoContainerProperty, TString> properties;
    properties[EPortoContainerProperty::Command] = "cmd";
    properties[EPortoContainerProperty::MemoryLimit] = "ml";

    auto test = [&properties](const TPortoGetResponse& ulimitResponse) {
        struct TMyPortoClient : public TMockPortoClient {
            TMyPortoClient(const TPortoGetResponse& ulimitResponse)
                : UlimitResponse_(ulimitResponse)
            {}

            TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
                const TVector<TPortoContainerName>& names
                , const TVector<EPortoContainerProperty>& props
            ) override {
                ++Calls_;
                LastNames_ = names;
                LastProps_ = props;

                TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;
                result[names[0]] = TMap<EPortoContainerProperty, TPortoGetResponse>();
                result[names[0]][EPortoContainerProperty::Command] = CreateTPortoGetResponse("cmd", 0, "");
                result[names[0]][EPortoContainerProperty::MemoryLimit] = CreateTPortoGetResponse("ml", 0, "");
                result[names[0]][EPortoContainerProperty::Private] = CreateTPortoGetResponse(PackContainerPrivate({CP_READY, "tree_hash"}), 0, "");
                result[names[0]][EPortoContainerProperty::Ulimit] = UlimitResponse_;

                return result;
            }

            size_t Calls_ = 0;
            TVector<TPortoContainerName> LastNames_;
            TVector<EPortoContainerProperty> LastProps_;
            TPortoGetResponse UlimitResponse_;
        };

        TPortoClientPtr porto = new TMyPortoClient(ulimitResponse);
        const TString workloadId = "my_workload";
        TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
        statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
        TPortoContainerName containerName = {"some_name"};
        auto node = CreateNode(porto, statusRepository, workloadId, containerName, "tree_hash", properties, {});
        auto result = node->Tick(MockTickContext(logger));

        UNIT_ASSERT_C(result, result.Error().Message);
        UNIT_ASSERT_EQUAL_C(ENodeStatus::SUCCESS, result.Success().Status, result.Success().Message);

        UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
        UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->LastNames_.size());
        UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)porto.Get())->LastNames_[0]);
        UNIT_ASSERT_EQUAL(4, ((TMyPortoClient*)porto.Get())->LastProps_.size());
    };

    TPortoGetResponse testResponse;

    // test with not empty Ulimit
    properties[EPortoContainerProperty::Ulimit] = "core: 0 unlimited; ";
    testResponse = CreateTPortoGetResponse("core: 0 unlimited; ", 0, "");
    test(testResponse);

    // test with empty Ulimit
    properties[EPortoContainerProperty::Ulimit] = "core: ; ";
    testResponse = CreateTPortoGetResponse("", 0, "");
    test(testResponse);
}

Y_UNIT_TEST(TestGetAndCheckPropertyNoContainer) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& names
            , const TVector<EPortoContainerProperty>& props
        ) override {
            ++Calls_;
            LastNames_ = names;
            LastProps_ = props;

            TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;

            return result;
        }

        size_t Calls_ = 0;
        TVector<TPortoContainerName> LastNames_;
        TVector<EPortoContainerProperty> LastProps_;
    };

    TPortoClientPtr porto = new TMyPortoClient();
    const TString workloadId = "my_workload";
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TPortoContainerName containerName = {"some_name"};

    TMap<EPortoContainerProperty, TString> properties;
    properties[EPortoContainerProperty::Command] = "cmd";
    properties[EPortoContainerProperty::MemoryLimit] = "ml";

    auto node = CreateNode(porto, statusRepository, workloadId, containerName, "tree_hash", properties, {});
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "doesn't contain");

    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->LastNames_.size());
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)porto.Get())->LastNames_[0]);
    UNIT_ASSERT_EQUAL(3, ((TMyPortoClient*)porto.Get())->LastProps_.size());
}

Y_UNIT_TEST(TestGetAndCheckPropertyGetResponseError) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& names
            , const TVector<EPortoContainerProperty>& props
        ) override {
            ++Calls_;
            LastNames_ = names;
            LastProps_ = props;

            TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;
            result[names[0]] = TMap<EPortoContainerProperty, TPortoGetResponse>();
            result[names[0]][EPortoContainerProperty::Command] = CreateTPortoGetResponse("cmd", 1, "ErrorMsg");
            result[names[0]][EPortoContainerProperty::MemoryLimit] = CreateTPortoGetResponse("ml", 1, "");
            result[names[0]][EPortoContainerProperty::Private] = CreateTPortoGetResponse(PackContainerPrivate({CP_READY, "tree_hash"}), 1, "");

            return result;
        }

        size_t Calls_ = 0;
        TVector<TPortoContainerName> LastNames_;
        TVector<EPortoContainerProperty> LastProps_;
    };

    TPortoClientPtr porto = new TMyPortoClient();
    const TString workloadId = "my_workload";
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TPortoContainerName containerName = {"some_name"};

    TMap<EPortoContainerProperty, TString> properties;
    properties[EPortoContainerProperty::Command] = "cmd";
    properties[EPortoContainerProperty::MemoryLimit] = "ml";

    auto node = CreateNode(porto, statusRepository, workloadId, containerName, "tree_hash", properties, {});
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "ErrorMsg");

    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->LastNames_.size());
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)porto.Get())->LastNames_[0]);
    UNIT_ASSERT_EQUAL(3, ((TMyPortoClient*)porto.Get())->LastProps_.size());
}

Y_UNIT_TEST(TestGetAndCheckPropertyCheckFail) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& names
            , const TVector<EPortoContainerProperty>& props
        ) override {
            ++Calls_;
            LastNames_ = names;
            LastProps_ = props;

            TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;
            result[names[0]] = TMap<EPortoContainerProperty, TPortoGetResponse>();
            result[names[0]][EPortoContainerProperty::Command] = CreateTPortoGetResponse("cmd", 0, "");
            result[names[0]][EPortoContainerProperty::MemoryLimit] = CreateTPortoGetResponse("other_ml", 0, "");
            result[names[0]][EPortoContainerProperty::Private] = CreateTPortoGetResponse(PackContainerPrivate({CP_READY, "tree_hash"}),  0, "");

            return result;
        }

        size_t Calls_ = 0;
        TVector<TPortoContainerName> LastNames_;
        TVector<EPortoContainerProperty> LastProps_;
    };

    TPortoClientPtr porto = new TMyPortoClient();
    const TString workloadId = "my_workload";
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TPortoContainerName containerName = {"some_name"};

    TMap<EPortoContainerProperty, TString> properties;
    properties[EPortoContainerProperty::Command] = "cmd";
    properties[EPortoContainerProperty::MemoryLimit] = "ml";

    auto node = CreateNode(porto, statusRepository, workloadId, containerName, "tree_hash", properties, {});
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);

    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->LastNames_.size());
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)porto.Get())->LastNames_[0]);
    UNIT_ASSERT_EQUAL(3, ((TMyPortoClient*)porto.Get())->LastProps_.size());
}

Y_UNIT_TEST(TestGetAndCheckPropertyWrongHash) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& names
            , const TVector<EPortoContainerProperty>& props
        ) override {
            ++Calls_;
            LastNames_ = names;
            LastProps_ = props;

            TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;
            result[names[0]] = TMap<EPortoContainerProperty, TPortoGetResponse>();
            result[names[0]][EPortoContainerProperty::Command] = CreateTPortoGetResponse("cmd", 0, "");
            result[names[0]][EPortoContainerProperty::MemoryLimit] = CreateTPortoGetResponse("ml", 0, "");
            result[names[0]][EPortoContainerProperty::Private] = CreateTPortoGetResponse(PackContainerPrivate({CP_READY, "wrong_hash"}), 0, "");

            return result;
        }

        size_t Calls_ = 0;
        TVector<TPortoContainerName> LastNames_;
        TVector<EPortoContainerProperty> LastProps_;
    };

    TPortoClientPtr porto = new TMyPortoClient();
    const TString workloadId = "my_workload";
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TPortoContainerName containerName = {"some_name"};

    TMap<EPortoContainerProperty, TString> properties;
    properties[EPortoContainerProperty::Command] = "cmd";
    properties[EPortoContainerProperty::MemoryLimit] = "ml";

    auto node = CreateNode(porto, statusRepository, workloadId, containerName, "tree_hash", properties, {});
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);

    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->LastNames_.size());
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)porto.Get())->LastNames_[0]);
    UNIT_ASSERT_EQUAL(3, ((TMyPortoClient*)porto.Get())->LastProps_.size());
}

Y_UNIT_TEST(TestGetAndCheckPropertyWithFailingPortoNoContainer) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& names
            , const TVector<EPortoContainerProperty>& props
        ) override {
            ++Calls_;
            LastNames_ = names;
            LastProps_ = props;

            return TPortoError{EPortoError::ContainerDoesNotExist, "Get", "NO"};
        }

        size_t Calls_ = 0;
        TVector<TPortoContainerName> LastNames_;
        TVector<EPortoContainerProperty> LastProps_;
    };

    TPortoClientPtr porto = new TMyPortoFailingClient();
    const TString workloadId = "my_workload";
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TPortoContainerName containerName = {"some_name"};

    TMap<EPortoContainerProperty, TString> properties;
    properties[EPortoContainerProperty::Command] = "cmd";
    properties[EPortoContainerProperty::MemoryLimit] = "ml";

    auto node = CreateNode(porto, statusRepository, workloadId, containerName, "tree_hash", properties, {});
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "Get");

    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->LastNames_.size());
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoFailingClient*)porto.Get())->LastNames_[0]);
    UNIT_ASSERT_EQUAL(3, ((TMyPortoFailingClient*)porto.Get())->LastProps_.size());
}

Y_UNIT_TEST(TestGetAndCheckPropertyWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& names
            , const TVector<EPortoContainerProperty>& props
        ) override {
            ++Calls_;
            LastNames_ = names;
            LastProps_ = props;

            return TPortoError{EPortoError::Unknown, "Get", "NO"};
        }

        size_t Calls_ = 0;
        TVector<TPortoContainerName> LastNames_;
        TVector<EPortoContainerProperty> LastProps_;
    };

    TPortoClientPtr porto = new TMyPortoFailingClient();
    const TString workloadId = "my_workload";
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TPortoContainerName containerName = {"some_name"};

    TMap<EPortoContainerProperty, TString> properties;
    properties[EPortoContainerProperty::Command] = "cmd";
    properties[EPortoContainerProperty::MemoryLimit] = "ml";

    auto node = CreateNode(porto, statusRepository, workloadId, containerName, "tree_hash", properties, {});
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "Get");

    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->LastNames_.size());
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoFailingClient*)porto.Get())->LastNames_[0]);
    UNIT_ASSERT_EQUAL(3, ((TMyPortoFailingClient*)porto.Get())->LastProps_.size());
}

Y_UNIT_TEST(TestGetAndCheckPropertyWithFailingPortoProperty) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(
            const TVector<TPortoContainerName>& names
            , const TVector<EPortoContainerProperty>& props
        ) override {
            ++Calls_;
            LastNames_ = names;
            LastProps_ = props;

            TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;
            result[names[0]] = TMap<EPortoContainerProperty, TPortoGetResponse>();
            result[names[0]][EPortoContainerProperty::Command] = CreateTPortoGetResponse("cmd", EPortoError::ContainerDoesNotExist, "ErrorMsg");

            return result;
        }

        size_t Calls_ = 0;
        TVector<TPortoContainerName> LastNames_;
        TVector<EPortoContainerProperty> LastProps_;
    };

    TPortoClientPtr porto = new TMyPortoFailingClient();
    const TString workloadId = "my_workload";
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
    TPortoContainerName containerName = {"some_name"};

    TMap<EPortoContainerProperty, TString> properties;
    properties[EPortoContainerProperty::Command] = "cmd";

    auto node = CreateNode(porto, statusRepository, workloadId, containerName, "tree_hash", properties, {});
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "does not exist");

    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->LastNames_.size());
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoFailingClient*)porto.Get())->LastNames_[0]);
    UNIT_ASSERT_EQUAL(2, ((TMyPortoFailingClient*)porto.Get())->LastProps_.size());
}

Y_UNIT_TEST(TestSpecialCompare) {
    auto node = CreateNode(new TMockPortoClient(), new TWorkloadStatusRepository(), "my_workload", TPortoContainerName::NoEscape("some_name"), "tree_hash", {}, {EPortoContainerProperty::Env});

    auto isSuccess = [](const TExpected<void, TTickResult>& result) {
        return (bool)result;
    };

    auto isFailure = [](const TExpected<void, TTickResult>& result) {
        return !(bool)result && (bool)result.Error() && result.Error().Success().Status == ENodeStatus::FAILURE;
    };

    auto isError = [](const TExpected<void, TTickResult>& result) {
        return !(bool)result && !(bool)result.Error();
    };

    {
        EPortoContainerProperty property = EPortoContainerProperty::CpuLimit;

        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(property, "", "0c")));
        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(property, "0c", "0c")));
        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(property, "0.1234567890778c", "0.1239435363445c")));

        UNIT_ASSERT(isFailure(node->ComparePropertyValues(property, "", "0.1c")));
        UNIT_ASSERT(isFailure(node->ComparePropertyValues(property, "0.2c", "0.22c")));

        UNIT_ASSERT(isError(node->ComparePropertyValues(property, "0.2", "0.2"))); // wrong suffix
    }

    {
        EPortoContainerProperty property = EPortoContainerProperty::Cwd;

        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(property, "", "/some/cwd")));
        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(property, "/some/cwd", "/some/cwd")));

        UNIT_ASSERT(isFailure(node->ComparePropertyValues(property, "/some/cwd", "")));
        UNIT_ASSERT(isFailure(node->ComparePropertyValues(property, "/some/cwd", "/other/cwd")));
    }

    {
        EPortoContainerProperty property = EPortoContainerProperty::Command;

        auto result = node->ComparePropertyValues(property, "my_command1", "my_command2");
        UNIT_ASSERT(isFailure(result));
        UNIT_ASSERT_STRING_CONTAINS_C(result.Error().Success().Message, "my_command1", result.Error().Success().Message);
    }

    {
        EPortoContainerProperty property = EPortoContainerProperty::Env;

        auto result = node->ComparePropertyValues(property, "a=b", "a=b;c=d");
        UNIT_ASSERT(isFailure(result));
        UNIT_ASSERT_STRING_CONTAINS_C(result.Error().Success().Message, "<Value is hidden>", result.Error().Success().Message);
        UNIT_ASSERT_C(result.Error().Success().Message.find("a=b") == TString::npos, "Hidden value in result '" << result.Error().Success().Message << "'");
    }

    {
        EPortoContainerProperty property = EPortoContainerProperty::IoLimit;

        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(property, "", "")));
        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(property, ";;;    ;  \n\n", "")));
        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(property, "    /tmp r: 3;     /tmp w: 4    ", "/tmp w: 4;/tmp r: 3")));
        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(property, "/tmp\\; r: 3;/tmp w: 4", "/tmp w: 4;/tmp\\; r: 3")));
        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(property, ";;/tmp r: 3;;; ;;/tmp w: 4;;;  ;;", "/tmp w: 4;;/tmp r: 3")));

        UNIT_ASSERT(isFailure(node->ComparePropertyValues(property, "/tmp r: 3", "")));
        UNIT_ASSERT(isFailure(node->ComparePropertyValues(property, "  /tmp r: 3  ", "/tmp w: 4")));
        UNIT_ASSERT(isFailure(node->ComparePropertyValues(property, "/tmp; r: 3", "/tmp\\; r: 3")));
    }

    {
        std::random_device rd;
        std::mt19937 gen(rd());
        std::uniform_int_distribution<ui64> dist(10000000, 99999999);

        auto makeHiddenSecret = [&](const TString& secret) {
            auto sep = secret.find("=");
            auto key = secret.substr(0, sep);
            auto value = secret.substr(sep + 1);
            auto salt = ToString(dist(gen));
            return TStringBuilder() << key << "=<secret salt=" << salt << " md5=" << MD5::Calc(salt + value) << ">";
        };

        auto unionSecrets = [](const TVector<TString>& secrets, char delimiter = ';') {
            TStringBuilder builder;
            for (const auto& secret: secrets) {
                builder << secret << delimiter;
            }
            builder.pop_back();
            return (TString)builder;
        };

        EPortoContainerProperty property = EPortoContainerProperty::EnvSecret;

        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(property, "", "")));
        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(property, "key=value", makeHiddenSecret("key=value"))));
        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(
            property,
            "key1=value1;key2=value2",
            unionSecrets({makeHiddenSecret("key1=value1"), makeHiddenSecret("key2=value2")})
        )));
        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(
            property,
            "key1=value1;key2=value2",
            unionSecrets({makeHiddenSecret("key2=value2"), makeHiddenSecret("key1=value1")})
        )));
        UNIT_ASSERT(isSuccess(node->ComparePropertyValues(
            property,
            "key1 = val\\\\ue1 ; key2 = val\\;ue2 ",
            unionSecrets({makeHiddenSecret("key1=val\\ue1"), makeHiddenSecret("key2=val;ue2")})
        )));

        UNIT_ASSERT(isFailure(node->ComparePropertyValues(property, "key=value", makeHiddenSecret("key=value1"))));
        UNIT_ASSERT(isFailure(node->ComparePropertyValues(property, "key=value1", makeHiddenSecret("key=value"))));
        UNIT_ASSERT(isFailure(node->ComparePropertyValues(property, "key1=value1;key2=value2", makeHiddenSecret("key=value"))));
        UNIT_ASSERT(isFailure(node->ComparePropertyValues(
            property,
            "key=value",
            unionSecrets({makeHiddenSecret("key1=value1"), makeHiddenSecret("key2=value2")})
        )));
        UNIT_ASSERT(isFailure(node->ComparePropertyValues(
            property,
            "key1=value1;key2=value2",
            unionSecrets({makeHiddenSecret("key1=value1"), makeHiddenSecret("key3=value3")})
        )));

        // Wrong format
        UNIT_ASSERT(isError(node->ComparePropertyValues(property, "key=value", "key=<salt=1111 md5=1111>")));
        UNIT_ASSERT(isError(node->ComparePropertyValues(property, "key=value", "key=<secret salt=1111>")));
        UNIT_ASSERT(isError(node->ComparePropertyValues(property, "key=value", "key=<secret md5=1111>")));
        UNIT_ASSERT(isError(node->ComparePropertyValues(property, "key=value", "key=<secret salt=1111 md5=1111")));
        UNIT_ASSERT(isError(node->ComparePropertyValues(
            property,
            "key1=value1;key2=value2",
            unionSecrets({makeHiddenSecret("key1=value1)"), "key2"})
        )));
        UNIT_ASSERT(isError(node->ComparePropertyValues(
            property,
            "key1=value1;key2",
            unionSecrets({makeHiddenSecret("key1=value1"), makeHiddenSecret("key3=value3")})
        )));
    }
}

}

} // namespace NInfra::NPodAgent::NPortoTest::NTestPortoGetAndCheckPropertiesNode
