#include "porto_get_and_check_volume_properties_node.h"

namespace {

TString GetPortoVolumeAttribute(
    const NInfra::NPodAgent::TPortoVolume& portoVolume
    , const NInfra::NPodAgent::EPortoVolumeProperty attribute
) {
    switch (attribute) {
        case NInfra::NPodAgent::EPortoVolumeProperty::Private:
            return portoVolume.private_value();
        case NInfra::NPodAgent::EPortoVolumeProperty::Storage:
            return portoVolume.storage();
        case NInfra::NPodAgent::EPortoVolumeProperty::ReadOnly:
            return portoVolume.read_only() ? "true" : "false";
        case NInfra::NPodAgent::EPortoVolumeProperty::INodeUsed:
            return ToString(portoVolume.inodes().usage());
        case NInfra::NPodAgent::EPortoVolumeProperty::State:
            return portoVolume.state();

        case NInfra::NPodAgent::EPortoVolumeProperty::SpaceLimit:
        case NInfra::NPodAgent::EPortoVolumeProperty::INodeLimit:
        case NInfra::NPodAgent::EPortoVolumeProperty::SpaceGuarantee:
        case NInfra::NPodAgent::EPortoVolumeProperty::INodeGuarantee:
        case NInfra::NPodAgent::EPortoVolumeProperty::SpaceUsed:
        case NInfra::NPodAgent::EPortoVolumeProperty::SpaceAvailable:
        case NInfra::NPodAgent::EPortoVolumeProperty::INodeAvailable:
        case NInfra::NPodAgent::EPortoVolumeProperty::Containers:
        case NInfra::NPodAgent::EPortoVolumeProperty::TargetContainer:
        case NInfra::NPodAgent::EPortoVolumeProperty::Layers:
        case NInfra::NPodAgent::EPortoVolumeProperty::Backend:
        case NInfra::NPodAgent::EPortoVolumeProperty::Place:
            ythrow yexception() << ToString(attribute) << " shouldn't be provided";
    }
    ythrow yexception() << "Cannot parse type: " << ToString(attribute);
}

} // namespace

namespace NInfra::NPodAgent {

ENodeType TPortoGetAndCheckVolumePropertiesNode::GetType() const {
    return TPortoGetAndCheckVolumePropertiesNode::NODE_TYPE;
}

TExpected<TFuture<TExpected<TVector<TPortoVolume>, TPortoError>>, TTickResult> TPortoGetAndCheckVolumePropertiesNode::PortoCall(TTickContextPtr /*context*/) {
    return Porto_->ListVolumes(Path_);
}

TTickResult TPortoGetAndCheckVolumePropertiesNode::ProcessPortoResult(TTickContextPtr /*context*/, TExpected<TVector<TPortoVolume>, TPortoError>& result) {
    if (result) {
        if (result.Success().size() != 1) {
            return TNodeError{"only 1 volume expected, got " + ToString(result.Success().size())};
        }

        const TPortoVolume& volume = result.Success().front();
        if (volume.path() != Path_) {
            return TNodeError({TStringBuilder() << "porto ListVolumes call doesn't contain '" << Path_ << "' volume"});
        }

        for (const auto& [property, expectedValue] : Properties_) {
            if (const auto& actualValue = GetPortoVolumeAttribute(volume, property); actualValue != expectedValue) {
                return TNodeSuccess(
                    ENodeStatus::FAILURE
                    , TStringBuilder()
                        << "'" << property
                        << "'property expected '" << expectedValue
                        << "' got '" << actualValue << "'"
                );
            }
        }

        return TNodeSuccess(ENodeStatus::SUCCESS);
    } else {
        if (result.Error().Code == EPortoError::VolumeNotFound) {
            return TNodeSuccess(ENodeStatus::FAILURE, TStringBuilder() << "Volume '" << Path_ << "' not found");
        }
        return TNodeError({ToString(result.Error())});
    }
}

} // namespace NInfra::NPodAgent
