#include "porto_get_and_check_volume_properties_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoGetAndCheckVolumePropertiesNode {

Y_UNIT_TEST_SUITE(PortoGetAndCheckVolumePropertyNodeSuite) {

static TLogger logger({});

TPortoGetAndCheckVolumePropertiesNodePtr CreateNode(
    TPortoClientPtr porto
    , const TString& path
    , TMap<EPortoVolumeProperty, TString>& properties
) {
    return new TPortoGetAndCheckVolumePropertiesNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , new TAsyncPortoClient(porto, new TFakeThreadPool())
        , path
        , properties
    );
}

Y_UNIT_TEST(TestGetAndCheckProperty) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(
            const TString& path
            , const TPortoContainerName& container
        ) override {
            ++Calls_;
            LastPath_ = path;
            LastContainer_ = container;

            TVector<TPortoVolume> result(1);
            result[0].set_path(path);
            result[0].set_private_value("volume_hash");
            result[0].set_storage("my_storage");

            return result;
        }

        size_t Calls_ = 0;
        TString LastPath_;
        TPortoContainerName LastContainer_{""};
    };

    TPortoClientPtr porto = new TMyPortoClient();

    const TString path = "my_path";
    TMap<EPortoVolumeProperty, TString> properties;
    properties[EPortoVolumeProperty::Private] = "volume_hash";
    properties[EPortoVolumeProperty::Storage] = "my_storage";

    auto node = CreateNode(porto, path, properties);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(path, ((TMyPortoClient*)porto.Get())->LastPath_);
    UNIT_ASSERT_EQUAL(TPortoContainerName{""}, ((TMyPortoClient*)porto.Get())->LastContainer_);
}

Y_UNIT_TEST(TestGetAndCheckPropertyManyVolumes) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(
            const TString& path
            , const TPortoContainerName& container
        ) override {
            ++Calls_;
            LastPath_ = path;
            LastContainer_ = container;

            TVector<TPortoVolume> result(2);

            return result;
        }

        size_t Calls_ = 0;
        TString LastPath_;
        TPortoContainerName LastContainer_{""};
    };

    TPortoClientPtr porto = new TMyPortoClient();

    const TString path = "my_path";
    TMap<EPortoVolumeProperty, TString> properties;
    properties[EPortoVolumeProperty::Private] = "volume_hash";
    properties[EPortoVolumeProperty::Storage] = "my_storage";

    auto node = CreateNode(porto, path, properties);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "only 1 volume expected, got 2");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(path, ((TMyPortoClient*)porto.Get())->LastPath_);
    UNIT_ASSERT_EQUAL(TPortoContainerName{""}, ((TMyPortoClient*)porto.Get())->LastContainer_);
}

Y_UNIT_TEST(TestGetAndCheckPropertyFail) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(
            const TString& path
            , const TPortoContainerName& container
        ) override {
            ++Calls_;
            LastPath_ = path;
            LastContainer_ = container;

            TVector<TPortoVolume> result(1);
            result[0].set_path(path);
            result[0].set_private_value("volume_hash");
            result[0].set_storage("wrong");

            return result;
        }

        size_t Calls_ = 0;
        TString LastPath_;
        TPortoContainerName LastContainer_{""};
    };

    TPortoClientPtr porto = new TMyPortoClient();

    const TString path = "my_path";
    TMap<EPortoVolumeProperty, TString> properties;
    properties[EPortoVolumeProperty::Private] = "volume_hash";
    properties[EPortoVolumeProperty::Storage] = "my_storage";

    auto node = CreateNode(porto, path, properties);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(path, ((TMyPortoClient*)porto.Get())->LastPath_);
    UNIT_ASSERT_EQUAL(TPortoContainerName{""}, ((TMyPortoClient*)porto.Get())->LastContainer_);
}

Y_UNIT_TEST(TestGetAndCheckPropertyVolumeNotFound) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(
            const TString& path
            , const TPortoContainerName& container
        ) override {
            ++Calls_;
            LastPath_ = path;
            LastContainer_ = container;

            return TPortoError{EPortoError::VolumeNotFound, "", "" ,""};
        }

        size_t Calls_ = 0;
        TString LastPath_;
        TPortoContainerName LastContainer_{""};
    };

    TPortoClientPtr porto = new TMyPortoClient();

    const TString path = "my_path";
    TMap<EPortoVolumeProperty, TString> properties;
    properties[EPortoVolumeProperty::Private] = "volume_hash";
    properties[EPortoVolumeProperty::Storage] = "my_storage";

    auto node = CreateNode(porto, path, properties);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(path, ((TMyPortoClient*)porto.Get())->LastPath_);
    UNIT_ASSERT_EQUAL(TPortoContainerName{""}, ((TMyPortoClient*)porto.Get())->LastContainer_);
}

Y_UNIT_TEST(TestGetAndCheckPropertyNoVolume) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(
            const TString& path
            , const TPortoContainerName& container
        ) override {
            ++Calls_;
            LastPath_ = path;
            LastContainer_ = container;

            TVector<TPortoVolume> result(1);
            result[0].set_path("wrong_path");
            result[0].set_private_value("volume_hash");
            result[0].set_storage("my_storage");

            return result;
        }

        size_t Calls_ = 0;
        TString LastPath_;
        TPortoContainerName LastContainer_{""};
    };

    TPortoClientPtr porto = new TMyPortoClient();

    const TString path = "my_path";
    TMap<EPortoVolumeProperty, TString> properties;
    properties[EPortoVolumeProperty::Private] = "volume_hash";
    properties[EPortoVolumeProperty::Storage] = "my_storage";

    auto node = CreateNode(porto, path, properties);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "porto ListVolumes call doesn't contain 'my_path' volume");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(path, ((TMyPortoClient*)porto.Get())->LastPath_);
    UNIT_ASSERT_EQUAL(TPortoContainerName{""}, ((TMyPortoClient*)porto.Get())->LastContainer_);
}

Y_UNIT_TEST(TestIncorrectProperies) {
    TPortoClientPtr porto = new TMockPortoClient();

    const TString path = "my_path";
    TMap<EPortoVolumeProperty, TString> properties;
    properties[EPortoVolumeProperty::INodeUsed] = "3000";

    UNIT_ASSERT_EXCEPTION_CONTAINS(
        CreateNode(porto, path, properties)
        , yexception
        , "property 'inode_used' is not implemented"
    );
}

}

} // namespace NInfra::NPodAgent::NTestPortoGetAndCheckVolumePropertiesNode
