#include "porto_get_container_state_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoGetContainerStateNode {

Y_UNIT_TEST_SUITE(PortoGetContainerStateNodeSuite) {

static TLogger logger({});

TPortoGetContainerStateNodePtr CreateNode(TAsyncPortoClientPtr porto, const TPortoContainerName& name) {
    return new TPortoGetContainerStateNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , name
    );
}

Y_UNIT_TEST(TestGetStateSuccess) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;
            Property_ = property;

            return TString("value");
        }

        TPortoContainerName Name_ = TString("");
        EPortoContainerProperty Property_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};

    auto creator = CreateNode(porto, name);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL("value", result.Success().Message);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::State, ((TMyPortoClient*)myPorto.Get())->Property_);
}

Y_UNIT_TEST(TestGetStateFailure) {
    struct TMyPortoClientFails : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;
            Property_ = property;

            return TPortoError{EPortoError::ContainerDoesNotExist, "internal error", "NO"};
        }

        TPortoContainerName Name_ = TString("");
        EPortoContainerProperty Property_;
    };

    TPortoClientPtr myPorto = new TMyPortoClientFails();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};

    auto creator = CreateNode(porto, name);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "internal error");
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClientFails*)myPorto.Get())->Name_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::State, ((TMyPortoClientFails*)myPorto.Get())->Property_);
}

Y_UNIT_TEST(TestGetStateError) {
    struct TMyPortoClientFails : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;
            Property_ = property;

            return TPortoError{EPortoError::Unknown, "internal error", "NO"};
        }

        TPortoContainerName Name_ = TString("");
        EPortoContainerProperty Property_;
    };

    TPortoClientPtr myPorto = new TMyPortoClientFails();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};

    auto creator = CreateNode(porto, name);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "internal error");
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClientFails*)myPorto.Get())->Name_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::State, ((TMyPortoClientFails*)myPorto.Get())->Property_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoGetContainerStateNode
