#include "porto_get_download_progress_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/pod_agent/libs/pod_agent/status_repository/layer_status_repository.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NPortoGetDownloadProgressNodeTest {

Y_UNIT_TEST_SUITE(PortoGetDownloadProgressNodeSuite) {

static TLogger logger({});

TPortoGetDownloadProgressNodePtr CreateNode(
    TPortoClientPtr porto
    , const TPortoContainerName& containerName
    , TLayerStatusRepositoryPtr statusRepository
    , const TString& layerHash
    , bool needCheckDownloadProgress
) {
    return new TPortoGetDownloadProgressNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , new TAsyncPortoClient(porto, new TFakeThreadPool())
        , statusRepository
        , containerName
        , layerHash
        , needCheckDownloadProgress
    );
}

Y_UNIT_TEST(TestDownloadOkWithWget) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetStderr(const TPortoContainerName& name, int /* offset */, int /* length */, int /* flags */) override {
            Name_ = name;
            return TString("106500K .......... .......... .......... .......... .......... 62%  112M 1s");
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();

    TPortoContainerName name = {"container_0"};

    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    const TString layerHash = "layer_hash";
    const TString layerId = "layer_id";
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerHash));

    auto node = CreateNode(myPorto, name, statusRepository, layerHash, true);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);

    auto status = statusRepository->GetObjectStatus(layerId, nullptr);
    UNIT_ASSERT_EQUAL_C(status.download_progress().percent(), 62, status.download_progress().percent());
}

Y_UNIT_TEST(TestDownloadOkWithSkyget) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetStderr(const TPortoContainerName& name, int /* offset */, int /* length */, int /* flags */) override {
            Name_ = name;
            return TString("[-] Downloading [h:-1 s:-1 c:-1 t:-1 cc:-1 tq:-1 tw:-1 tc:-1 td:-1] [80.00MiB] [735.11 MB/s] || [ETA:  0:00:00]  47%");
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();

    TPortoContainerName name = {"container_0"};

    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    const TString layerHash = "layer_hash";
    const TString layerId = "layer_id";
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerHash));

    auto node = CreateNode(myPorto, name, statusRepository, layerHash, true);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);

    auto status = statusRepository->GetObjectStatus(layerId, nullptr);
    UNIT_ASSERT_EQUAL_C(status.download_progress().percent(), 47, status.download_progress().percent());
}

Y_UNIT_TEST(TestDownloadWithManyPercents) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetStderr(const TPortoContainerName& name, int /* offset */, int /* length */, int /* flags */) override {
            Name_ = name;
            return TString(
                "29250K .......... .......... .......... .......... .......... 17% 98,7M 5s\n \
                106500K .......... .......... .......... .......... .......... 62%  112M 1s\n \
                117050K .......... .......... .......... .......... .......... 68%  106M 1s"
            );
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();

    TPortoContainerName name = {"container_0"};

    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    const TString layerHash = "layer_hash";
    const TString layerId = "layer_id";
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerHash));

    auto node = CreateNode(myPorto, name, statusRepository, layerHash, true);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);

    auto status = statusRepository->GetObjectStatus(layerId, nullptr);
    UNIT_ASSERT_EQUAL_C(status.download_progress().percent(), 68, status.download_progress().percent());
}

Y_UNIT_TEST(TestDownloadWithoutPercents) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetStderr(const TPortoContainerName& name, int /* offset */, int /* length */, int /* flags */) override {
            Name_ = name;

            return TString(
                "--2019-08-22 12:11:58--  https://proxy.sandbox.yandex-team.ru/514352150\n\
                Resolving proxy.sandbox.yandex-team.ru (proxy.sandbox.yandex-team.ru)... 2a02:6b8:0:3400::1:6, 5.255.240.6\n\
                Connecting to proxy.sandbox.yandex-team.ru (proxy.sandbox.yandex-team.ru)|2a02:6b8:0:3400::1:6|:443... connected.\n\
                HTTP request sent, awaiting response... 200 OK\n\
                Length: 175893824 (168M) [application/x-tar]\n\
                Saving to: ‘514352150’"
            );
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();

    TPortoContainerName name = {"container_0"};

    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    const TString layerHash = "layer_hash";
    const TString layerId = "layer_id";
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerHash));

    auto node = CreateNode(myPorto, name, statusRepository, layerHash, true);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);

    auto status = statusRepository->GetObjectStatus(layerId, nullptr);
    UNIT_ASSERT_EQUAL_C(status.download_progress().percent(), 0, status.download_progress().percent());
}

Y_UNIT_TEST(TestPercentInName) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetStderr(const TPortoContainerName& name, int /* offset */, int /* length */, int /* flags */) override {
            Name_ = name;
            return TString("2019-08-22 12:12:01 (51,1 MB/s) - ‘514352150%’ saved [175893824/175893824]");
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();

    TPortoContainerName name = {"container_0"};

    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    const TString layerHash = "layer_hash";
    const TString layerId = "layer_id";
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerHash));

    auto node = CreateNode(myPorto, name, statusRepository, layerHash, true);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);

    auto status = statusRepository->GetObjectStatus(layerId, nullptr);
    UNIT_ASSERT_EQUAL_C(status.download_progress().percent(), 0, status.download_progress().percent());
}

Y_UNIT_TEST(TestAnotherPercentInName) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetStderr(const TPortoContainerName& name, int /* offset */, int /* length */, int /* flags */) override {
            Name_ = name;
            return TString("2019-08-22 12:12:01 (51,1 MB/s) - ‘fifty_seven%’ saved [175893824/175893824]");
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();

    TPortoContainerName name = {"container_0"};

    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    const TString layerHash = "layer_hash";
    const TString layerId = "layer_id";
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerHash));

    auto node = CreateNode(myPorto, name, statusRepository, layerHash, true);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);

    auto status = statusRepository->GetObjectStatus(layerId, nullptr);
    UNIT_ASSERT_EQUAL_C(status.download_progress().percent(), 0, status.download_progress().percent());
}

Y_UNIT_TEST(TestNoNeedToCheckProgress) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetStderr(const TPortoContainerName& name, int /* offset */, int /* length */, int /* flags */) override {
            Name_ = name;
            return TString("106500K .......... .......... .......... .......... .......... 62%  112M 1s");
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();

    TPortoContainerName name = {"container_0"};

    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    const TString layerHash = "layer_hash";
    const TString layerId = "layer_id";
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerHash));

    auto node = CreateNode(myPorto, name, statusRepository, layerHash, false);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);

    auto status = statusRepository->GetObjectStatus(layerId, nullptr);
    UNIT_ASSERT_EQUAL_C(status.download_progress().percent(), 0, status.download_progress().percent());
}

}

} // namespace NInfra::NPodAgent::NPortoGetDownloadProgressNodeTest
