#include "porto_get_private_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoGetPrivateNode {

Y_UNIT_TEST_SUITE(TPortoGetPrivateNodeSuite) {

static TLogger logger({});

TPortoGetContainerPrivateNodePtr CreateNode(TAsyncPortoClientPtr porto, const TPortoContainerName& name, const TString& treeHash) {
    return new TPortoGetContainerPrivateNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , name
        , ""
        , treeHash
    );
}

Y_UNIT_TEST(TestGetContainerPrivate) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;
            Property_ = property;

            return TString("ready:tree_hash");
        }

        TPortoContainerName Name_ = TString("");
        EPortoContainerProperty Property_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};

    auto creator = CreateNode(porto, name, "tree_hash");
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(ToString(CP_READY), result.Success().Message);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::Private, ((TMyPortoClient*)myPorto.Get())->Property_);
}

Y_UNIT_TEST(TestGetLayerPrivate) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetLayerPrivate(const TString& layer, const TString& place) override {
            Layer_ = layer;
            Place_ = place;

            return TString("cleaned:tree_hash");
        }

        TString Layer_ = "";
        TString Place_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString layer = "my_layer";
    const TString place = "/some_place";

    TPortoGetLayerPrivateNodePtr node = new TPortoGetLayerPrivateNode(TBasicTreeNodeDescriptor{1, "title"}, porto, layer, place, "tree_hash");
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(ToString(LP_CLEANED), result.Success().Message);
    UNIT_ASSERT_EQUAL(layer, ((TMyPortoClient*)myPorto.Get())->Layer_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoClient*)myPorto.Get())->Place_);
}

Y_UNIT_TEST(TestContainerDoesNotExist) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;
            Property_ = property;

            TPortoError error;
            error.Code = EPortoError::ContainerDoesNotExist;

            return error;
        }

        TPortoContainerName Name_ = TString("");
        EPortoContainerProperty Property_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};

    auto creator = CreateNode(porto, name, "tree_hash");
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(ToString(CP_EXTRACTION_FAILURE), result.Success().Message);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::Private, ((TMyPortoClient*)myPorto.Get())->Property_);
}

Y_UNIT_TEST(TestPrivateWrongHash) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;
            Property_ = property;

            return TString("ready:other_tree_hash");
        }

        TPortoContainerName Name_ = TString("");
        EPortoContainerProperty Property_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TPortoContainerName name = {"container_0"};

    auto creator = CreateNode(porto, name, "tree_hash");
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(ToString(CP_EXTRACTION_FAILURE), result.Success().Message);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)myPorto.Get())->Name_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::Private, ((TMyPortoClient*)myPorto.Get())->Property_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoGetPrivateNode
