#include "porto_import_layer_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoImportLayerNode {

Y_UNIT_TEST_SUITE(PortoImportLayerNodeSuite) {

static TLogger logger({});

Y_UNIT_TEST(TestImportLayer) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> ImportLayer(const TString& layer, const TString& tarball, bool /*merge*/, const TString& place, const TString& privateValue) override {
            ++Calls_;
            LastLayer_ = layer;
            LastTarball_ = tarball;
            LastPlace_ = place;
            LastPrivate_ = privateValue;

            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        size_t Calls_ = 0;
        TString LastLayer_ = "";
        TString LastTarball_ = "";
        TString LastPlace_ = "";
        TString LastPrivate_ = "";
    };

    TString layerPath = "/some/path/layer.tar.xz";
    TString layerPlace = "/some_place";
    TString treeHash = "layer_tree_hash";

    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString layerId = "some_layer_id";
    const TString layerDownloadHash = "some_layer_download_hash";
    const TString layerName = "some_layer_name";
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerDownloadHash));

    TPortoImportLayerNodePtr node = new TPortoImportLayerNode(TBasicTreeNodeDescriptor{1, "title"}, porto, statusRepository, layerDownloadHash, layerName, layerPlace, layerPath, treeHash);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(0, statusRepository->GetObjectStatus(layerId).fail_counter());
    UNIT_ASSERT_EQUAL("", statusRepository->GetObjectStatus(layerId).failed().message());
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(layerName, ((TMyPortoClient*)myPorto.Get())->LastLayer_);
    UNIT_ASSERT_EQUAL(layerPath, ((TMyPortoClient*)myPorto.Get())->LastTarball_);
    UNIT_ASSERT_EQUAL(layerPlace, ((TMyPortoClient*)myPorto.Get())->LastPlace_);
    UNIT_ASSERT_EQUAL(PackContainerPrivate({CP_EMPTY, treeHash}), ((TMyPortoClient*)myPorto.Get())->LastPrivate_);
}

Y_UNIT_TEST(TestImportLayerWithFailingImport) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> ImportLayer(const TString& layer, const TString& tarball, bool , const TString& place, const TString& privateValue) override {
            ++Calls_;
            LastLayer_ = layer;
            LastTarball_ = tarball;
            LastPlace_ = place;
            LastPrivate_ = privateValue;

            return TPortoError{EPortoError::LayerAlreadyExists, "ImportLayer", "NO"};
        }

        size_t Calls_ = 0;
        TString LastLayer_ = "";
        TString LastTarball_ = "";
        TString LastPlace_ = "";
        TString LastPrivate_ = "";
    };

    TString layerPath = "/some/path/layer.tar.xz";
    TString layerPlace = "/some_place";
    TString treeHash = "layer_tree_hash";

    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString layerId = "some_layer_id";
    const TString layerDownloadHash = "some_layer_download_hash";
    const TString layerName = "some_layer_name";
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerDownloadHash));

    TPortoImportLayerNodePtr node = new TPortoImportLayerNode(TBasicTreeNodeDescriptor{1, "title"}, porto, statusRepository, layerDownloadHash, layerName, layerPlace, layerPath, treeHash);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "ImportLayer");
    UNIT_ASSERT_EQUAL(1, statusRepository->GetObjectStatus(layerId).fail_counter());
    UNIT_ASSERT_EQUAL(result.Success().Message, statusRepository->GetObjectStatus(layerId).failed().message());
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(layerName, ((TMyPortoFailingClient*)myPorto.Get())->LastLayer_);
    UNIT_ASSERT_EQUAL(layerPath, ((TMyPortoFailingClient*)myPorto.Get())->LastTarball_);
    UNIT_ASSERT_EQUAL(layerPlace, ((TMyPortoFailingClient*)myPorto.Get())->LastPlace_);
    UNIT_ASSERT_EQUAL(PackContainerPrivate({CP_EMPTY, treeHash}), ((TMyPortoFailingClient*)myPorto.Get())->LastPrivate_);
}

Y_UNIT_TEST(TestImportLayerWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> ImportLayer(const TString& layer, const TString& tarball, bool , const TString& place, const TString& privateValue) override {
            ++Calls_;
            LastLayer_ = layer;
            LastTarball_ = tarball;
            LastPlace_ = place;
            LastPrivate_ = privateValue;

            return TPortoError{EPortoError::Unknown, "ImportLayer", "NO"};
        }

        size_t Calls_ = 0;
        TString LastLayer_ = "";
        TString LastTarball_ = "";
        TString LastPlace_ = "";
        TString LastPrivate_ = "";
    };

    TString layerPath = "/some/path/layer.tar.xz";
    TString layerPlace = "/some_place";
    TString treeHash = "layer_tree_hash";

    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString layerId = "some_layer_id";
    const TString layerDownloadHash = "some_layer_download_hash";
    const TString layerName = "some_layer_name";
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerDownloadHash));

    TPortoImportLayerNodePtr node = new TPortoImportLayerNode(TBasicTreeNodeDescriptor{1, "title"}, porto, statusRepository, layerDownloadHash, layerName, layerPlace, layerPath, treeHash);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "ImportLayer");
    UNIT_ASSERT_EQUAL(1, statusRepository->GetObjectStatus(layerId).fail_counter());
    UNIT_ASSERT_EQUAL(result.Error().Message, statusRepository->GetObjectStatus(layerId).failed().message());
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(layerName, ((TMyPortoFailingClient*)myPorto.Get())->LastLayer_);
    UNIT_ASSERT_EQUAL(layerPath, ((TMyPortoFailingClient*)myPorto.Get())->LastTarball_);
    UNIT_ASSERT_EQUAL(layerPlace, ((TMyPortoFailingClient*)myPorto.Get())->LastPlace_);
    UNIT_ASSERT_EQUAL(PackContainerPrivate({CP_EMPTY, treeHash}), ((TMyPortoFailingClient*)myPorto.Get())->LastPrivate_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoImportLayerNode
