#include "porto_kill_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoKillNode {

Y_UNIT_TEST_SUITE(PortoKillNodeSuite) {

static TLogger logger({});

TPortoKillNodePtr PortoCreator(
    TPortoClientPtr porto
    , TWorkloadStatusRepositoryPtr statusRepository
    , const NStatusRepositoryTypes::TContainerDescription& container
    , const TPortoContainerName& containerName
) {
    return new TPortoKillNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , new TAsyncPortoClient(porto, new TFakeThreadPool())
        , statusRepository
        , container
        , containerName
    );
}

Y_UNIT_TEST(TestKillContainer) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int) override {
            ++Calls_;
            LastName_ = name;

            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
    };

    const TString workloadId = "my_workload";
    TPortoClientPtr porto = new TMyPortoClient();
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    NStatusRepositoryTypes::TContainerDescription container(
        workloadId
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::START
    );
    TPortoContainerName containerName = {"some_name"};
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto creator = PortoCreator(porto, statusRepository, container, containerName);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().current().fail_reason(), "");
    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().system_failure_counter(), 0);

    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)porto.Get())->LastName_);
}

Y_UNIT_TEST(TestKillContainerWithFailingKill) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int) override {
            ++Calls_;
            LastName_ = name;

            return TPortoError{EPortoError::InvalidValue, "Kill", "NO"};
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
    };

    TPortoClientPtr porto = new TMyPortoFailingClient();

    const TString workloadId = "my_workload";
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    NStatusRepositoryTypes::TContainerDescription container(
        workloadId
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::START
    );
    TPortoContainerName containerName = {"some_name"};
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto creator = PortoCreator(porto, statusRepository, container, containerName);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "Kill");

    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().current().fail_reason(), result.Success().Message);
    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().current().state(), API::EContainerState_UNKNOWN);
    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().system_failure_counter(), 1);

    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoFailingClient*)porto.Get())->LastName_);
}

Y_UNIT_TEST(TestKillContainerWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int) override {
            ++Calls_;
            LastName_ = name;

            return TPortoError{EPortoError::Unknown, "Kill", "NO"};
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
    };

    TPortoClientPtr porto = new TMyPortoFailingClient();

    const TString workloadId = "my_workload";
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    NStatusRepositoryTypes::TContainerDescription container(
        workloadId
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::START
    );
    TPortoContainerName containerName = {"some_name"};
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto creator = PortoCreator(porto, statusRepository, container, containerName);
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "Kill");

    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().current().fail_reason(), result.Error().Message);
    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().current().state(), API::EContainerState_UNKNOWN);
    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().system_failure_counter(), 1);

    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoFailingClient*)porto.Get())->LastName_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoKillNode
