#include "porto_layer_exists_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoLayerExistsNode {

Y_UNIT_TEST_SUITE(PortoLayerExistsNodeSuite) {

static TLogger logger({});

Y_UNIT_TEST(TestLayerExists) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoLayer>, TPortoError> ListLayers(const TString& place, const TString& mask) override {
            ++Calls_;
            Place_ = place;
            Mask_ = mask;

            TPortoLayer layer;
            layer.set_name(Mask_);

            return TVector<TPortoLayer>({layer});
        }

        size_t Calls_ = 0;
        TString Place_ = "";
        TString Mask_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString layer = "some_layer";
    const TString place = "/some_place";

    TPortoLayerExistsNodePtr node = new TPortoLayerExistsNode(TBasicTreeNodeDescriptor{1, "title"}, porto, layer, place);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoClient*)myPorto.Get())->Place_);
    UNIT_ASSERT_EQUAL(layer, ((TMyPortoClient*)myPorto.Get())->Mask_);
}

Y_UNIT_TEST(TestLayerNotExists) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoLayer>, TPortoError> ListLayers(const TString& place, const TString& mask) override {
            ++Calls_;
            Place_ = place;
            Mask_ = mask;

            return TVector<TPortoLayer>();
        }

        size_t Calls_ = 0;
        TString Place_ = "";
        TString Mask_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString layer = "some_layer";
    const TString place = "/some_place";

    TPortoLayerExistsNodePtr node = new TPortoLayerExistsNode(TBasicTreeNodeDescriptor{1, "title"}, porto, layer, place);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoClient*)myPorto.Get())->Place_);
    UNIT_ASSERT_EQUAL(layer, ((TMyPortoClient*)myPorto.Get())->Mask_);
}

Y_UNIT_TEST(TestLayerExistsWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<TVector<TPortoLayer>, TPortoError> ListLayers(const TString& place, const TString& mask) override {
            ++Calls_;
            Place_ = place;
            Mask_ = mask;

            return TPortoError{EPortoError::Unknown, "ListVolumes", "NO"};
        }

        size_t Calls_ = 0;
        TString Place_ = "";
        TString Mask_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString layer = "some_layer";
    const TString place = "/some_place";

    TPortoLayerExistsNodePtr node = new TPortoLayerExistsNode(TBasicTreeNodeDescriptor{1, "title"}, porto, layer, place);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "ListVolumes");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoFailingClient*)myPorto.Get())->Place_);
    UNIT_ASSERT_EQUAL(layer, ((TMyPortoFailingClient*)myPorto.Get())->Mask_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoLayerExistsNode
